// ============================================================
// DBAOps Sentinel - REST API Server
// Your Database Guardian - Backend Services
// ============================================================

const express = require('express');
const sql = require('mssql');
const cors = require('cors');
const helmet = require('helmet');
const compression = require('compression');
const jwt = require('jsonwebtoken');
const bcrypt = require('bcryptjs');
require('dotenv').config();

const app = express();
const PORT = process.env.PORT || 3000;

// ============================================================
// MIDDLEWARE
// ============================================================
app.use(helmet());
app.use(compression());
app.use(cors());
app.use(express.json());
app.use(express.static('../'));

// Request logging
app.use((req, res, next) => {
    console.log(`[${new Date().toISOString()}] ${req.method} ${req.path}`);
    next();
});

// ============================================================
// DATABASE CONFIGURATION
// ============================================================
const dbConfig = {
    server: process.env.DB_SERVER || 'JBGWALLY\\MSSQLSERVER2022',
    database: process.env.DB_NAME || 'DBAOpsRepository',
    options: {
        encrypt: false,
        trustServerCertificate: true,
        enableArithAbort: true
    },
    // Use Windows Authentication
    authentication: {
        type: 'ntlm',
        options: {
            domain: process.env.DB_DOMAIN || '',
            userName: process.env.DB_USER || '',
            password: process.env.DB_PASSWORD || ''
        }
    },
    pool: {
        max: 10,
        min: 0,
        idleTimeoutMillis: 30000
    }
};

// Alternative: SQL Authentication
const dbConfigSQL = {
    server: process.env.DB_SERVER || 'JBGWALLY\\MSSQLSERVER2022',
    database: process.env.DB_NAME || 'DBAOpsRepository',
    user: process.env.DB_USER || 'sentinel_api',
    password: process.env.DB_PASSWORD || '',
    options: {
        encrypt: false,
        trustServerCertificate: true,
        enableArithAbort: true
    },
    pool: {
        max: 10,
        min: 0,
        idleTimeoutMillis: 30000
    }
};

// Global connection pool
let pool = null;

async function getPool() {
    if (!pool) {
        try {
            // Try Windows Auth first, fall back to SQL Auth
            pool = await sql.connect(dbConfig);
            console.log('✅ Connected to SQL Server (Windows Auth)');
        } catch (err) {
            console.log('⚠️ Windows Auth failed, trying SQL Auth...');
            try {
                pool = await sql.connect(dbConfigSQL);
                console.log('✅ Connected to SQL Server (SQL Auth)');
            } catch (err2) {
                console.error('❌ Database connection failed:', err2.message);
                throw err2;
            }
        }
    }
    return pool;
}

// ============================================================
// HEALTH CHECK ENDPOINT
// ============================================================
app.get('/api/health', async (req, res) => {
    try {
        const pool = await getPool();
        const result = await pool.request().query('SELECT 1 as status, GETDATE() as serverTime, @@VERSION as version');
        res.json({
            status: 'healthy',
            api: 'DBAOps Sentinel API v2.4.0',
            database: 'connected',
            serverTime: result.recordset[0].serverTime,
            sqlVersion: result.recordset[0].version.split('\n')[0]
        });
    } catch (err) {
        res.status(500).json({
            status: 'unhealthy',
            error: err.message
        });
    }
});

// ============================================================
// DASHBOARD OVERVIEW
// ============================================================
app.get('/api/dashboard/overview', async (req, res) => {
    try {
        const pool = await getPool();
        
        // Get server count
        const servers = await pool.request().query(`
            SELECT COUNT(*) as count FROM inventory.SQLInstances WHERE IsActive = 1
        `);
        
        // Get database count
        const databases = await pool.request().query(`
            SELECT COUNT(*) as count FROM inventory.Databases WHERE IsActive = 1
        `);
        
        // Get active alerts
        const alerts = await pool.request().query(`
            SELECT COUNT(*) as count FROM monitoring.Alerts 
            WHERE Status IN ('Open', 'Acknowledged') AND IsActive = 1
        `);
        
        // Get auto-healed count (this month)
        const healed = await pool.request().query(`
            SELECT COUNT(*) as count FROM sentinel.HealingLog 
            WHERE ExecutedAt >= DATEADD(MONTH, DATEDIFF(MONTH, 0, GETDATE()), 0)
            AND Success = 1
        `);
        
        // Get security score
        const security = await pool.request().query(`
            SELECT AVG(CAST(Score as FLOAT)) as avgScore FROM security.AssessmentResults
            WHERE AssessedAt >= DATEADD(DAY, -7, GETDATE())
        `);
        
        // Get compliance score
        const compliance = await pool.request().query(`
            SELECT TOP 1 CompliancePercentage as score 
            FROM compliance.AssessmentSummary 
            ORDER BY AssessedAt DESC
        `);

        res.json({
            servers: servers.recordset[0]?.count || 0,
            databases: databases.recordset[0]?.count || 0,
            activeAlerts: alerts.recordset[0]?.count || 0,
            autoHealed: healed.recordset[0]?.count || 0,
            securityScore: Math.round(security.recordset[0]?.avgScore || 0),
            complianceScore: compliance.recordset[0]?.score || 0,
            lastUpdated: new Date().toISOString()
        });
    } catch (err) {
        console.error('Dashboard overview error:', err);
        res.status(500).json({ error: err.message });
    }
});

// ============================================================
// SERVERS ENDPOINTS
// ============================================================
app.get('/api/servers', async (req, res) => {
    try {
        const pool = await getPool();
        const result = await pool.request().query(`
            SELECT 
                i.InstanceID as id,
                i.ServerName as serverName,
                i.InstanceName as instanceName,
                CASE WHEN i.InstanceName = 'MSSQLSERVER' 
                     THEN i.ServerName 
                     ELSE i.ServerName + '\\' + i.InstanceName 
                END as fullName,
                i.SQLVersion as version,
                i.Edition as edition,
                i.Environment as environment,
                i.IsOnline as isOnline,
                i.LastSeen as lastSeen,
                i.IsSentinelProtected as sentinelProtected,
                m.CPUPercent as cpu,
                m.MemoryPercent as memory,
                m.DiskPercent as disk,
                m.HealthScore as health,
                (SELECT COUNT(*) FROM inventory.Databases d WHERE d.InstanceID = i.InstanceID AND d.IsActive = 1) as databaseCount
            FROM inventory.SQLInstances i
            LEFT JOIN monitoring.LatestMetrics m ON i.InstanceID = m.InstanceID
            WHERE i.IsActive = 1
            ORDER BY i.ServerName, i.InstanceName
        `);
        
        res.json({
            servers: result.recordset.map(s => ({
                id: s.id,
                name: s.fullName,
                serverName: s.serverName,
                instanceName: s.instanceName,
                version: s.version,
                edition: s.edition,
                environment: s.environment,
                status: s.isOnline ? 'online' : 'offline',
                lastSeen: s.lastSeen,
                sentinel: s.sentinelProtected,
                cpu: s.cpu || 0,
                memory: s.memory || 0,
                disk: s.disk || 0,
                health: s.health || 0,
                databases: s.databaseCount
            })),
            total: result.recordset.length,
            timestamp: new Date().toISOString()
        });
    } catch (err) {
        console.error('Servers error:', err);
        res.status(500).json({ error: err.message });
    }
});

app.get('/api/servers/:id', async (req, res) => {
    try {
        const pool = await getPool();
        const result = await pool.request()
            .input('id', sql.Int, req.params.id)
            .query(`
                SELECT 
                    i.*,
                    m.CPUPercent, m.MemoryPercent, m.DiskPercent, m.HealthScore,
                    m.ConnectionCount, m.BatchRequestsPerSec, m.PageLifeExpectancy
                FROM inventory.SQLInstances i
                LEFT JOIN monitoring.LatestMetrics m ON i.InstanceID = m.InstanceID
                WHERE i.InstanceID = @id
            `);
        
        if (result.recordset.length === 0) {
            return res.status(404).json({ error: 'Server not found' });
        }
        
        res.json(result.recordset[0]);
    } catch (err) {
        res.status(500).json({ error: err.message });
    }
});

// ============================================================
// DATABASES ENDPOINTS
// ============================================================
app.get('/api/databases', async (req, res) => {
    try {
        const pool = await getPool();
        const result = await pool.request().query(`
            SELECT 
                d.DatabaseID as id,
                d.DatabaseName as name,
                i.ServerName + CASE WHEN i.InstanceName != 'MSSQLSERVER' 
                    THEN '\\' + i.InstanceName ELSE '' END as serverName,
                d.SizeMB as sizeMB,
                d.RecoveryModel as recoveryModel,
                d.CompatibilityLevel as compatLevel,
                d.State as state,
                d.IsEncrypted as encrypted,
                d.LastFullBackup as lastBackup,
                d.CreatedAt as created
            FROM inventory.Databases d
            JOIN inventory.SQLInstances i ON d.InstanceID = i.InstanceID
            WHERE d.IsActive = 1
            ORDER BY i.ServerName, d.DatabaseName
        `);
        
        res.json({
            databases: result.recordset,
            total: result.recordset.length
        });
    } catch (err) {
        res.status(500).json({ error: err.message });
    }
});

// ============================================================
// ALERTS ENDPOINTS
// ============================================================
app.get('/api/alerts', async (req, res) => {
    try {
        const pool = await getPool();
        const status = req.query.status || 'Open,Acknowledged';
        
        const result = await pool.request().query(`
            SELECT 
                a.AlertID as id,
                a.AlertCode as code,
                a.AlertType as type,
                a.Severity as severity,
                a.Title as title,
                a.Message as message,
                a.ServerName as serverName,
                a.DatabaseName as databaseName,
                a.Status as status,
                a.CreatedAt as createdAt,
                a.AcknowledgedAt as acknowledgedAt,
                a.ResolvedAt as resolvedAt,
                a.CanAutoHeal as canHeal,
                a.HealingRuleID as healingRule
            FROM monitoring.Alerts a
            WHERE a.Status IN (${status.split(',').map(s => `'${s.trim()}'`).join(',')})
            AND a.IsActive = 1
            ORDER BY 
                CASE a.Severity 
                    WHEN 'Critical' THEN 1 
                    WHEN 'High' THEN 2 
                    WHEN 'Medium' THEN 3 
                    ELSE 4 
                END,
                a.CreatedAt DESC
        `);
        
        res.json({
            alerts: result.recordset,
            total: result.recordset.length
        });
    } catch (err) {
        res.status(500).json({ error: err.message });
    }
});

app.post('/api/alerts/:id/acknowledge', async (req, res) => {
    try {
        const pool = await getPool();
        await pool.request()
            .input('id', sql.Int, req.params.id)
            .input('user', sql.NVarChar, req.body.user || 'API')
            .query(`
                UPDATE monitoring.Alerts 
                SET Status = 'Acknowledged', 
                    AcknowledgedAt = GETDATE(),
                    AcknowledgedBy = @user
                WHERE AlertID = @id
            `);
        
        res.json({ success: true, message: 'Alert acknowledged' });
    } catch (err) {
        res.status(500).json({ error: err.message });
    }
});

app.post('/api/alerts/:id/heal', async (req, res) => {
    try {
        const pool = await getPool();
        
        // Get alert details
        const alert = await pool.request()
            .input('id', sql.Int, req.params.id)
            .query(`SELECT * FROM monitoring.Alerts WHERE AlertID = @id`);
        
        if (alert.recordset.length === 0) {
            return res.status(404).json({ error: 'Alert not found' });
        }
        
        // Execute healing (this would call the actual healing procedure)
        await pool.request()
            .input('alertId', sql.Int, req.params.id)
            .input('user', sql.NVarChar, req.body.user || 'Sentinel')
            .execute('sentinel.ExecuteHealing');
        
        res.json({ success: true, message: 'Sentinel healing initiated' });
    } catch (err) {
        res.status(500).json({ error: err.message });
    }
});

// ============================================================
// HEALTH METRICS ENDPOINTS
// ============================================================
app.get('/api/metrics/health', async (req, res) => {
    try {
        const pool = await getPool();
        const result = await pool.request().query(`
            SELECT 
                AVG(CPUPercent) as avgCPU,
                AVG(MemoryPercent) as avgMemory,
                AVG(DiskPercent) as avgDisk,
                AVG(NetworkPercent) as avgNetwork,
                AVG(HealthScore) as avgHealth,
                MAX(CPUPercent) as maxCPU,
                MAX(MemoryPercent) as maxMemory,
                MAX(DiskPercent) as maxDisk
            FROM monitoring.LatestMetrics m
            JOIN inventory.SQLInstances i ON m.InstanceID = i.InstanceID
            WHERE i.IsActive = 1 AND i.IsOnline = 1
        `);
        
        const metrics = result.recordset[0];
        res.json({
            cpu: Math.round(metrics.avgCPU || 0),
            memory: Math.round(metrics.avgMemory || 0),
            disk: Math.round(metrics.avgDisk || 0),
            network: Math.round(metrics.avgNetwork || 0),
            health: Math.round(metrics.avgHealth || 0),
            peaks: {
                cpu: Math.round(metrics.maxCPU || 0),
                memory: Math.round(metrics.maxMemory || 0),
                disk: Math.round(metrics.maxDisk || 0)
            }
        });
    } catch (err) {
        res.status(500).json({ error: err.message });
    }
});

app.get('/api/metrics/history/:metric', async (req, res) => {
    try {
        const pool = await getPool();
        const metric = req.params.metric;
        const hours = parseInt(req.query.hours) || 24;
        
        const columnMap = {
            cpu: 'CPUPercent',
            memory: 'MemoryPercent',
            disk: 'DiskPercent',
            network: 'NetworkPercent',
            health: 'HealthScore'
        };
        
        const column = columnMap[metric];
        if (!column) {
            return res.status(400).json({ error: 'Invalid metric' });
        }
        
        const result = await pool.request()
            .input('hours', sql.Int, hours)
            .query(`
                SELECT 
                    DATEADD(HOUR, DATEDIFF(HOUR, 0, CollectedAt), 0) as hour,
                    AVG(${column}) as value
                FROM monitoring.MetricsHistory
                WHERE CollectedAt >= DATEADD(HOUR, -@hours, GETDATE())
                GROUP BY DATEADD(HOUR, DATEDIFF(HOUR, 0, CollectedAt), 0)
                ORDER BY hour
            `);
        
        res.json({
            metric,
            hours,
            data: result.recordset
        });
    } catch (err) {
        res.status(500).json({ error: err.message });
    }
});

// ============================================================
// COMPLIANCE ENDPOINTS
// ============================================================
app.get('/api/compliance', async (req, res) => {
    try {
        const pool = await getPool();
        const result = await pool.request().query(`
            SELECT 
                c.FrameworkCode as code,
                c.FrameworkName as name,
                c.Icon as icon,
                c.TotalControls as totalControls,
                c.PassedControls as passedControls,
                CAST(c.PassedControls as FLOAT) / NULLIF(c.TotalControls, 0) * 100 as score,
                c.LastAssessment as lastAssessed,
                c.NextAssessment as nextAssessment,
                c.IsActive as isActive
            FROM compliance.FrameworkStatus c
            WHERE c.IsActive = 1
            ORDER BY c.SortOrder
        `);
        
        res.json({
            frameworks: result.recordset.map(f => ({
                ...f,
                score: Math.round(f.score || 0)
            })),
            total: result.recordset.length
        });
    } catch (err) {
        res.status(500).json({ error: err.message });
    }
});

app.get('/api/compliance/:framework', async (req, res) => {
    try {
        const pool = await getPool();
        const result = await pool.request()
            .input('framework', sql.NVarChar, req.params.framework)
            .query(`
                SELECT 
                    c.ControlID as id,
                    c.ControlCode as code,
                    c.ControlName as name,
                    c.Description as description,
                    c.Category as category,
                    c.Status as status,
                    c.LastChecked as lastChecked,
                    c.Evidence as evidence,
                    c.Remediation as remediation
                FROM compliance.Controls c
                WHERE c.FrameworkCode = @framework
                ORDER BY c.ControlCode
            `);
        
        res.json({
            framework: req.params.framework,
            controls: result.recordset,
            total: result.recordset.length,
            passed: result.recordset.filter(c => c.status === 'Pass').length
        });
    } catch (err) {
        res.status(500).json({ error: err.message });
    }
});

// ============================================================
// SECURITY ENDPOINTS
// ============================================================
app.get('/api/security/overview', async (req, res) => {
    try {
        const pool = await getPool();
        
        const score = await pool.request().query(`
            SELECT AVG(CAST(Score as FLOAT)) as avgScore 
            FROM security.AssessmentResults 
            WHERE AssessedAt >= DATEADD(DAY, -7, GETDATE())
        `);
        
        const tde = await pool.request().query(`
            SELECT 
                COUNT(CASE WHEN IsEncrypted = 1 THEN 1 END) as encrypted,
                COUNT(*) as total
            FROM inventory.Databases WHERE IsActive = 1
        `);
        
        const tls = await pool.request().query(`
            SELECT 
                COUNT(CASE WHEN TLSVersion >= '1.2' THEN 1 END) as compliant,
                COUNT(*) as total
            FROM inventory.SQLInstances WHERE IsActive = 1
        `);
        
        const sa = await pool.request().query(`
            SELECT COUNT(*) as count 
            FROM security.SAAccounts 
            WHERE IsEnabled = 1
        `);
        
        const overPrivileged = await pool.request().query(`
            SELECT COUNT(*) as count 
            FROM security.OverPrivilegedAccounts 
            WHERE IsActive = 1
        `);
        
        res.json({
            securityScore: Math.round(score.recordset[0]?.avgScore || 0),
            tde: {
                encrypted: tde.recordset[0]?.encrypted || 0,
                total: tde.recordset[0]?.total || 0,
                percentage: Math.round((tde.recordset[0]?.encrypted / tde.recordset[0]?.total) * 100) || 0
            },
            tls: {
                compliant: tls.recordset[0]?.compliant || 0,
                total: tls.recordset[0]?.total || 0,
                percentage: Math.round((tls.recordset[0]?.compliant / tls.recordset[0]?.total) * 100) || 0
            },
            saEnabled: sa.recordset[0]?.count || 0,
            overPrivileged: overPrivileged.recordset[0]?.count || 0
        });
    } catch (err) {
        res.status(500).json({ error: err.message });
    }
});

// ============================================================
// SENTINEL HEALING ENDPOINTS
// ============================================================
app.get('/api/sentinel/rules', async (req, res) => {
    try {
        const pool = await getPool();
        const result = await pool.request().query(`
            SELECT 
                r.RuleID as id,
                r.RuleName as name,
                r.Description as description,
                r.TriggerCondition as trigger,
                r.HealingAction as action,
                r.IsEnabled as enabled,
                r.ExecutionCount as executions,
                r.LastExecuted as lastExecuted,
                r.SuccessRate as successRate
            FROM sentinel.HealingRules r
            WHERE r.IsActive = 1
            ORDER BY r.ExecutionCount DESC
        `);
        
        res.json({
            rules: result.recordset,
            total: result.recordset.length
        });
    } catch (err) {
        res.status(500).json({ error: err.message });
    }
});

app.get('/api/sentinel/activity', async (req, res) => {
    try {
        const pool = await getPool();
        const limit = parseInt(req.query.limit) || 20;
        
        const result = await pool.request()
            .input('limit', sql.Int, limit)
            .query(`
                SELECT TOP (@limit)
                    l.LogID as id,
                    l.RuleName as rule,
                    l.ServerName as server,
                    l.Action as action,
                    l.Success as success,
                    l.Message as message,
                    l.ExecutedAt as timestamp
                FROM sentinel.HealingLog l
                ORDER BY l.ExecutedAt DESC
            `);
        
        res.json({
            activity: result.recordset,
            total: result.recordset.length
        });
    } catch (err) {
        res.status(500).json({ error: err.message });
    }
});

app.post('/api/sentinel/scan', async (req, res) => {
    try {
        const pool = await getPool();
        
        // Execute Sentinel scan procedure
        await pool.request().execute('sentinel.RunFullScan');
        
        res.json({ 
            success: true, 
            message: 'Sentinel scan initiated',
            timestamp: new Date().toISOString()
        });
    } catch (err) {
        res.status(500).json({ error: err.message });
    }
});

// ============================================================
// BACKUP ENDPOINTS
// ============================================================
app.get('/api/backups/status', async (req, res) => {
    try {
        const pool = await getPool();
        const result = await pool.request().query(`
            SELECT 
                d.DatabaseName as database,
                i.ServerName as server,
                b.BackupType as type,
                b.BackupSize as sizeMB,
                b.IsEncrypted as encrypted,
                b.CompletedAt as completedAt,
                b.Duration as durationSeconds,
                DATEDIFF(HOUR, b.CompletedAt, GETDATE()) as hoursAgo
            FROM backup.LatestBackups b
            JOIN inventory.Databases d ON b.DatabaseID = d.DatabaseID
            JOIN inventory.SQLInstances i ON d.InstanceID = i.InstanceID
            WHERE d.IsActive = 1
            ORDER BY b.CompletedAt DESC
        `);
        
        // Get overdue backups
        const overdue = await pool.request().query(`
            SELECT COUNT(*) as count
            FROM inventory.Databases d
            WHERE d.IsActive = 1
            AND (d.LastFullBackup IS NULL OR d.LastFullBackup < DATEADD(DAY, -1, GETDATE()))
        `);
        
        res.json({
            backups: result.recordset,
            overdueCount: overdue.recordset[0]?.count || 0
        });
    } catch (err) {
        res.status(500).json({ error: err.message });
    }
});

app.post('/api/backups/run', async (req, res) => {
    try {
        const pool = await getPool();
        const { serverId, databaseId, type } = req.body;
        
        await pool.request()
            .input('serverId', sql.Int, serverId)
            .input('databaseId', sql.Int, databaseId)
            .input('backupType', sql.NVarChar, type || 'Full')
            .execute('backup.InitiateBackup');
        
        res.json({ 
            success: true, 
            message: 'Backup initiated'
        });
    } catch (err) {
        res.status(500).json({ error: err.message });
    }
});

// ============================================================
// LICENSING ENDPOINTS
// ============================================================
app.get('/api/license', async (req, res) => {
    try {
        const pool = await getPool();
        const result = await pool.request().query(`
            SELECT TOP 1
                l.LicenseKey as licenseKey,
                l.CustomerName as customer,
                t.TierCode as tier,
                t.TierName as tierName,
                l.CurrentServers as serversUsed,
                t.MaxServers as serversMax,
                l.CurrentDatabases as databasesUsed,
                t.MaxDatabases as databasesMax,
                l.ExpirationDate as expires,
                l.Status as status
            FROM licensing.CustomerLicenses l
            JOIN licensing.Tiers t ON l.TierID = t.TierID
            WHERE l.Status = 'Active'
            ORDER BY l.PurchaseDate DESC
        `);
        
        res.json(result.recordset[0] || {
            tier: 'FREE',
            tierName: 'Free',
            serversUsed: 0,
            serversMax: 2,
            status: 'Active'
        });
    } catch (err) {
        res.status(500).json({ error: err.message });
    }
});

app.get('/api/license/industries', async (req, res) => {
    try {
        const pool = await getPool();
        const result = await pool.request().query(`
            SELECT 
                i.IndustryCode as code,
                i.IndustryName as name,
                i.Icon as icon,
                i.ComplianceFramework as framework,
                i.TotalControls as controls,
                i.MonthlyAddonPrice as addonPrice,
                CASE 
                    WHEN tia.AccessType = 'Included' THEN 'included'
                    WHEN tia.AccessType = 'Addon' THEN 'addon'
                    ELSE 'locked'
                END as status
            FROM licensing.IndustryModules i
            LEFT JOIN licensing.TierIndustryAccess tia ON i.IndustryID = tia.IndustryID
            LEFT JOIN licensing.CustomerLicenses cl ON tia.TierID = cl.TierID AND cl.Status = 'Active'
            WHERE i.IsActive = 1
            ORDER BY i.SortOrder
        `);
        
        res.json({
            industries: result.recordset
        });
    } catch (err) {
        res.status(500).json({ error: err.message });
    }
});

// ============================================================
// REPORTS ENDPOINTS
// ============================================================
app.post('/api/reports/generate', async (req, res) => {
    try {
        const pool = await getPool();
        const { reportType, format, dateRange } = req.body;
        
        // Log report request
        await pool.request()
            .input('reportType', sql.NVarChar, reportType)
            .input('format', sql.NVarChar, format || 'PDF')
            .input('requestedBy', sql.NVarChar, req.body.user || 'API')
            .execute('reports.GenerateReport');
        
        res.json({
            success: true,
            message: `${reportType} report generation started`,
            estimatedTime: '30 seconds'
        });
    } catch (err) {
        res.status(500).json({ error: err.message });
    }
});

// ============================================================
// DISCOVERY ENDPOINTS
// ============================================================
app.post('/api/discovery/scan', async (req, res) => {
    try {
        const pool = await getPool();
        const { method, target } = req.body;
        
        await pool.request()
            .input('method', sql.NVarChar, method || 'network')
            .input('target', sql.NVarChar, target || '')
            .execute('discovery.RunScan');
        
        res.json({
            success: true,
            message: 'Discovery scan initiated'
        });
    } catch (err) {
        res.status(500).json({ error: err.message });
    }
});

// ============================================================
// ERROR HANDLING
// ============================================================
app.use((err, req, res, next) => {
    console.error('Unhandled error:', err);
    res.status(500).json({
        error: 'Internal server error',
        message: err.message
    });
});

// 404 handler
app.use((req, res) => {
    res.status(404).json({ error: 'Endpoint not found' });
});

// ============================================================
// START SERVER
// ============================================================
app.listen(PORT, () => {
    console.log('');
    console.log('='.repeat(60));
    console.log('  🛡️  DBAOps Sentinel API Server');
    console.log('  Your Database Guardian - Backend Services');
    console.log('='.repeat(60));
    console.log(`  🚀 Server running on http://localhost:${PORT}`);
    console.log(`  📊 API endpoints: http://localhost:${PORT}/api/`);
    console.log(`  🏥 Health check: http://localhost:${PORT}/api/health`);
    console.log('='.repeat(60));
    console.log('');
});

module.exports = app;
