#Requires -RunAsAdministrator
<#
.SYNOPSIS
    DBAOps Sentinel Installer
.DESCRIPTION
    Installs DBAOps Sentinel - Your Database Guardian
.PARAMETER InstallPath
    Installation directory (default: C:\Program Files\DBAOps\Sentinel)
.PARAMETER LicenseKey
    License key for activation
.PARAMETER Trial
    Start 90-day trial
.PARAMETER Silent
    Silent installation
.EXAMPLE
    .\Install-Sentinel.ps1 -Trial
    .\Install-Sentinel.ps1 -LicenseKey "SENT-XXXX-XXXX-XXXX-XXXX"
#>

param(
    [string]$InstallPath = "C:\Program Files\DBAOps\Sentinel",
    [string]$LicenseKey,
    [switch]$Trial,
    [switch]$Silent
)

$ErrorActionPreference = "Stop"
$Version = "2.4.0"
$SourcePath = $PSScriptRoot

# Banner
function Show-Banner {
    Clear-Host
    Write-Host ""
    Write-Host "  ╔══════════════════════════════════════════════════════════════╗" -ForegroundColor Cyan
    Write-Host "  ║                                                              ║" -ForegroundColor Cyan
    Write-Host "  ║     ███████╗███████╗███╗   ██╗████████╗██╗███╗   ██╗███████╗ ║" -ForegroundColor Cyan
    Write-Host "  ║     ██╔════╝██╔════╝████╗  ██║╚══██╔══╝██║████╗  ██║██╔════╝ ║" -ForegroundColor Cyan
    Write-Host "  ║     ███████╗█████╗  ██╔██╗ ██║   ██║   ██║██╔██╗ ██║█████╗   ║" -ForegroundColor Cyan
    Write-Host "  ║     ╚════██║██╔══╝  ██║╚██╗██║   ██║   ██║██║╚██╗██║██╔══╝   ║" -ForegroundColor Cyan
    Write-Host "  ║     ███████║███████╗██║ ╚████║   ██║   ██║██║ ╚████║███████╗ ║" -ForegroundColor Cyan
    Write-Host "  ║     ╚══════╝╚══════╝╚═╝  ╚═══╝   ╚═╝   ╚═╝╚═╝  ╚═══╝╚══════╝ ║" -ForegroundColor Cyan
    Write-Host "  ║                                                              ║" -ForegroundColor Cyan
    Write-Host "  ║              DBAOps Sentinel v$Version Installer               ║" -ForegroundColor Cyan
    Write-Host "  ║                  Your Database Guardian                      ║" -ForegroundColor Cyan
    Write-Host "  ║                                                              ║" -ForegroundColor Cyan
    Write-Host "  ╚══════════════════════════════════════════════════════════════╝" -ForegroundColor Cyan
    Write-Host ""
}

# Check prerequisites
function Test-Prerequisites {
    Write-Host "  [1/7] Checking prerequisites..." -ForegroundColor White
    
    # PowerShell version
    if ($PSVersionTable.PSVersion.Major -lt 5) {
        throw "PowerShell 5.1 or higher required. Current: $($PSVersionTable.PSVersion)"
    }
    Write-Host "        PowerShell $($PSVersionTable.PSVersion)" -ForegroundColor Green
    
    # Admin rights
    $isAdmin = ([Security.Principal.WindowsPrincipal][Security.Principal.WindowsIdentity]::GetCurrent()).IsInRole([Security.Principal.WindowsBuiltInRole]::Administrator)
    if (-not $isAdmin) { throw "Administrator rights required" }
    Write-Host "        Administrator rights confirmed" -ForegroundColor Green
    
    # .NET Framework
    $dotnet = Get-ItemProperty "HKLM:\SOFTWARE\Microsoft\NET Framework Setup\NDP\v4\Full" -ErrorAction SilentlyContinue
    if ($dotnet.Release -ge 461808) {
        Write-Host "        .NET Framework 4.7.2+" -ForegroundColor Green
    } else {
        Write-Host "        .NET Framework 4.7.2+ recommended" -ForegroundColor Yellow
    }
    
    # Disk space
    $drive = (Split-Path $InstallPath -Qualifier)
    $freeSpace = (Get-PSDrive ($drive -replace ":","")).Free / 1GB
    if ($freeSpace -lt 1) { throw "Insufficient disk space. Need at least 1GB free." }
    Write-Host "        Disk space: $([math]::Round($freeSpace, 2)) GB free" -ForegroundColor Green
}

# Create directories
function Initialize-Directories {
    Write-Host "  [2/7] Creating directories..." -ForegroundColor White
    
    $dirs = @(
        $InstallPath,
        "$InstallPath\Config",
        "$InstallPath\Scripts",
        "$InstallPath\Schema",
        "$InstallPath\Dashboard",
        "$InstallPath\API",
        "$InstallPath\Logs",
        "$InstallPath\Data"
    )
    
    foreach ($dir in $dirs) {
        if (!(Test-Path $dir)) {
            New-Item -ItemType Directory -Path $dir -Force | Out-Null
        }
    }
    Write-Host "        Created installation directories" -ForegroundColor Green
}

# Copy files
function Copy-Files {
    Write-Host "  [3/7] Copying files..." -ForegroundColor White
    
    $folders = @("Config", "Scripts", "Schema", "Dashboard", "API", "Docs")
    
    foreach ($folder in $folders) {
        $src = Join-Path $SourcePath $folder
        $dst = Join-Path $InstallPath $folder
        if (Test-Path $src) {
            Copy-Item -Path "$src\*" -Destination $dst -Recurse -Force -ErrorAction SilentlyContinue
            Write-Host "        Copied $folder" -ForegroundColor Green
        }
    }
    
    # Copy root scripts
    Copy-Item "$SourcePath\*.ps1" -Destination $InstallPath -Force -ErrorAction SilentlyContinue
    Copy-Item "$SourcePath\*.md" -Destination $InstallPath -Force -ErrorAction SilentlyContinue
}

# Create configuration
function Initialize-Configuration {
    Write-Host "  [4/7] Creating configuration..." -ForegroundColor White
    
    $config = @{
        Sentinel = @{
            Version = $Version
            InstallPath = $InstallPath
            InstallDate = (Get-Date).ToString("yyyy-MM-dd HH:mm:ss")
        }
        License = @{
            Key = ""
            Type = "Scout"
            Activated = $false
            ExpiresAt = $null
        }
        Database = @{
            RepositoryServer = "localhost"
            RepositoryDatabase = "DBAOpsRepository"
            IntegratedSecurity = $true
            ConnectionTimeout = 30
        }
        Monitoring = @{
            Enabled = $true
            IntervalSeconds = 60
            RetentionDays = 7
        }
        SelfHealing = @{
            Enabled = $false
            RequiresLicense = "Guard"
            AutoApprove = $false
        }
        Alerts = @{
            Email = @{
                Enabled = $false
                SmtpServer = ""
                SmtpPort = 587
                UseSsl = $true
                From = ""
                To = @()
            }
            Slack = @{
                Enabled = $false
                WebhookUrl = ""
            }
        }
        Logging = @{
            Level = "Information"
            Path = "$InstallPath\Logs"
            RetentionDays = 30
        }
        API = @{
            Enabled = $true
            Port = 3000
            EnableSwagger = $true
        }
    }
    
    $configPath = "$InstallPath\Config\sentinel-config.json"
    $config | ConvertTo-Json -Depth 10 | Out-File $configPath -Encoding UTF8 -Force
    Write-Host "        Configuration created" -ForegroundColor Green
}

# Register scheduled task
function Register-SentinelService {
    Write-Host "  [5/7] Registering Sentinel service..." -ForegroundColor White
    
    $taskName = "DBAOps Sentinel Agent"
    $scriptPath = "$InstallPath\Scripts\Sentinel-Agent.ps1"
    
    # Remove existing
    Unregister-ScheduledTask -TaskName $taskName -Confirm:$false -ErrorAction SilentlyContinue
    
    # Create new task
    $action = New-ScheduledTaskAction -Execute "powershell.exe" -Argument "-NoProfile -ExecutionPolicy Bypass -WindowStyle Hidden -File `"$scriptPath`""
    $trigger = New-ScheduledTaskTrigger -AtStartup
    $trigger2 = New-ScheduledTaskTrigger -Once -At (Get-Date).AddMinutes(1) -RepetitionInterval (New-TimeSpan -Minutes 1)
    $settings = New-ScheduledTaskSettingsSet -AllowStartIfOnBatteries -DontStopIfGoingOnBatteries -StartWhenAvailable -RestartCount 3 -RestartInterval (New-TimeSpan -Minutes 1) -ExecutionTimeLimit (New-TimeSpan -Hours 0)
    $principal = New-ScheduledTaskPrincipal -UserId "SYSTEM" -LogonType ServiceAccount -RunLevel Highest
    
    Register-ScheduledTask -TaskName $taskName -Action $action -Trigger $trigger,$trigger2 -Settings $settings -Principal $principal -Description "DBAOps Sentinel - Your Database Guardian" | Out-Null
    Write-Host "        Sentinel Agent registered" -ForegroundColor Green
}

# Set environment variables
function Set-EnvironmentVariables {
    Write-Host "  [6/7] Setting environment variables..." -ForegroundColor White
    
    [Environment]::SetEnvironmentVariable("SENTINEL_HOME", $InstallPath, "Machine")
    [Environment]::SetEnvironmentVariable("SENTINEL_CONFIG", "$InstallPath\Config\sentinel-config.json", "Machine")
    
    # Add to PATH
    $path = [Environment]::GetEnvironmentVariable("PATH", "Machine")
    if ($path -notlike "*$InstallPath*") {
        [Environment]::SetEnvironmentVariable("PATH", "$path;$InstallPath;$InstallPath\Scripts", "Machine")
    }
    Write-Host "        Environment variables set" -ForegroundColor Green
}

# Activate license
function Set-License {
    Write-Host "  [7/7] Configuring license..." -ForegroundColor White
    
    $configPath = "$InstallPath\Config\sentinel-config.json"
    $config = Get-Content $configPath | ConvertFrom-Json
    
    if ($LicenseKey) {
        if ($LicenseKey -match "^SENT-[A-Z0-9]{4}-[A-Z0-9]{4}-[A-Z0-9]{4}-[A-Z0-9]{4}$") {
            # Determine tier from key prefix
            $tierCode = $LicenseKey.Substring(5, 1)
            $tier = switch ($tierCode) {
                'W' { "Watch" }
                'G' { "Guard" }
                'C' { "Command" }
                'F' { "Fortress" }
                'P' { "Government" }
                'V' { "Government" }
                default { "Watch" }
            }
            
            $config.License.Key = $LicenseKey
            $config.License.Type = $tier
            $config.License.Activated = $true
            
            # Check for perpetual
            if ($tierCode -eq 'P') {
                $config.License.ExpiresAt = "9999-12-31"
                Write-Host "        PERPETUAL license activated: $tier" -ForegroundColor Green
            } else {
                $config.License.ExpiresAt = (Get-Date).AddYears(1).ToString("yyyy-MM-dd")
                Write-Host "        License activated: $tier (expires: $($config.License.ExpiresAt))" -ForegroundColor Green
            }
        } else {
            Write-Host "        Invalid key format. Using Scout (Free)" -ForegroundColor Yellow
        }
    } elseif ($Trial) {
        $config.License.Type = "Trial"
        $config.License.Activated = $true
        $config.License.ExpiresAt = (Get-Date).AddDays(90).ToString("yyyy-MM-dd")
        $config.SelfHealing.Enabled = $true
        Write-Host "        90-day trial activated (Guard features)" -ForegroundColor Green
    } else {
        $config.License.Type = "Scout"
        Write-Host "        Scout (Free) edition - 2 servers max" -ForegroundColor Green
    }
    
    $config | ConvertTo-Json -Depth 10 | Out-File $configPath -Encoding UTF8 -Force
}

# Show completion
function Show-Complete {
    Write-Host ""
    Write-Host "  ╔══════════════════════════════════════════════════════════════╗" -ForegroundColor Green
    Write-Host "  ║                                                              ║" -ForegroundColor Green
    Write-Host "  ║           Installation Complete!                             ║" -ForegroundColor Green
    Write-Host "  ║                                                              ║" -ForegroundColor Green
    Write-Host "  ╚══════════════════════════════════════════════════════════════╝" -ForegroundColor Green
    Write-Host ""
    Write-Host "  Install Path:  $InstallPath" -ForegroundColor Cyan
    Write-Host "  Dashboard:     $InstallPath\Dashboard\index.html" -ForegroundColor Cyan
    Write-Host "  Config:        $InstallPath\Config\sentinel-config.json" -ForegroundColor Cyan
    Write-Host ""
    Write-Host "  Next Steps:" -ForegroundColor Yellow
    Write-Host "  1. Edit Config\sentinel-config.json with your SQL Server details"
    Write-Host "  2. Run: .\Deploy-Schema.ps1 to create the repository database"
    Write-Host "  3. Run: Start-ScheduledTask 'DBAOps Sentinel Agent'"
    Write-Host "  4. Open Dashboard\index.html in your browser"
    Write-Host ""
    Write-Host "  Support: support@dbaopsframework.com | (832) 314-3066" -ForegroundColor Cyan
    Write-Host ""
}

# Main
try {
    if (-not $Silent) { Show-Banner }
    Test-Prerequisites
    Initialize-Directories
    Copy-Files
    Initialize-Configuration
    Register-SentinelService
    Set-EnvironmentVariables
    Set-License
    if (-not $Silent) { Show-Complete }
    
    # Create uninstall registry entry
    $uninstallKey = "HKLM:\SOFTWARE\Microsoft\Windows\CurrentVersion\Uninstall\DBAOpsSentinel"
    if (!(Test-Path $uninstallKey)) { New-Item -Path $uninstallKey -Force | Out-Null }
    Set-ItemProperty -Path $uninstallKey -Name "DisplayName" -Value "DBAOps Sentinel"
    Set-ItemProperty -Path $uninstallKey -Name "DisplayVersion" -Value $Version
    Set-ItemProperty -Path $uninstallKey -Name "Publisher" -Value "RHOYLLE INNOVATION EMPIRE"
    Set-ItemProperty -Path $uninstallKey -Name "InstallLocation" -Value $InstallPath
    Set-ItemProperty -Path $uninstallKey -Name "UninstallString" -Value "powershell.exe -File `"$InstallPath\Uninstall-Sentinel.ps1`""
    
    exit 0
} catch {
    Write-Host ""
    Write-Host "  Installation failed: $_" -ForegroundColor Red
    Write-Host "  Contact: support@dbaopsframework.com" -ForegroundColor Yellow
    exit 1
}
