#Requires -RunAsAdministrator
<#
.SYNOPSIS
    Deploys DBAOps Sentinel database schema
.PARAMETER ServerInstance
    SQL Server instance (default: localhost)
.PARAMETER Database
    Repository database name (default: DBAOpsRepository)
#>

param(
    [string]$ServerInstance = "localhost",
    [string]$Database = "DBAOpsRepository",
    [switch]$CreateDatabase
)

$ErrorActionPreference = "Stop"
$SchemaPath = Join-Path $PSScriptRoot "Schema"

Write-Host ""
Write-Host "  DBAOps Sentinel - Schema Deployment" -ForegroundColor Cyan
Write-Host "  Server: $ServerInstance | Database: $Database" -ForegroundColor Cyan
Write-Host ""

# Check for sqlcmd or SqlServer module
$useSqlCmd = Get-Command sqlcmd -ErrorAction SilentlyContinue
$useSqlServer = Get-Module -ListAvailable SqlServer

function Invoke-SQL {
    param([string]$Query, [string]$Server = $ServerInstance, [string]$DB = $Database)
    
    if ($useSqlCmd) {
        $result = sqlcmd -S $Server -d $DB -Q $Query -h -1 -W 2>&1
        return $result
    } elseif ($useSqlServer) {
        Import-Module SqlServer
        return Invoke-Sqlcmd -ServerInstance $Server -Database $DB -Query $Query
    } else {
        # Use .NET
        $conn = New-Object System.Data.SqlClient.SqlConnection("Server=$Server;Database=$DB;Integrated Security=True;TrustServerCertificate=True")
        $conn.Open()
        $cmd = New-Object System.Data.SqlClient.SqlCommand($Query, $conn)
        $result = $cmd.ExecuteNonQuery()
        $conn.Close()
        return $result
    }
}

# Create database if needed
if ($CreateDatabase) {
    Write-Host "  [1/3] Creating database..." -ForegroundColor White
    try {
        $createDb = "IF NOT EXISTS (SELECT 1 FROM sys.databases WHERE name = '$Database') CREATE DATABASE [$Database]"
        Invoke-SQL -Query $createDb -DB "master"
        Write-Host "        Database created or already exists" -ForegroundColor Green
    } catch {
        Write-Host "        Error: $_" -ForegroundColor Red
    }
}

# Deploy schema files
Write-Host "  [2/3] Deploying schema files..." -ForegroundColor White

$schemaFiles = Get-ChildItem "$SchemaPath\*.sql" -ErrorAction SilentlyContinue | Sort-Object Name

if ($schemaFiles.Count -eq 0) {
    Write-Host "        No schema files found in $SchemaPath" -ForegroundColor Yellow
} else {
    foreach ($file in $schemaFiles) {
        Write-Host "        Deploying $($file.Name)..." -ForegroundColor Gray
        try {
            if ($useSqlCmd) {
                sqlcmd -S $ServerInstance -d $Database -i $file.FullName -b | Out-Null
            } else {
                $sql = Get-Content $file.FullName -Raw
                Invoke-SQL -Query $sql
            }
            Write-Host "        OK $($file.Name)" -ForegroundColor Green
        } catch {
            Write-Host "        FAILED $($file.Name): $_" -ForegroundColor Red
        }
    }
}

# Verify
Write-Host "  [3/3] Verifying deployment..." -ForegroundColor White
$schemas = Invoke-SQL -Query "SELECT name FROM sys.schemas WHERE name IN ('inventory','monitoring','sentinel','compliance','security','backup','licensing','reports')"
Write-Host "        Schemas deployed successfully" -ForegroundColor Green

Write-Host ""
Write-Host "  Schema deployment complete!" -ForegroundColor Green
Write-Host ""
