-- ============================================================
-- DBAOps: Discovery Mechanisms Schema
-- ============================================================
IF NOT EXISTS (SELECT 1 FROM sys.schemas WHERE name = 'discovery')
    EXEC('CREATE SCHEMA discovery');
GO

-- Discovery Methods Configuration
CREATE TABLE discovery.DiscoveryMethods (
    MethodID INT IDENTITY(1,1) PRIMARY KEY,
    MethodName NVARCHAR(50) NOT NULL,
    MethodType NVARCHAR(50) NOT NULL, -- Agent, APIConnector, NetworkScan, ManualRegistration
    Description NVARCHAR(500),
    IsEnabled BIT DEFAULT 1,
    Priority INT DEFAULT 100,
    ConfigurationJSON NVARCHAR(MAX),
    LastRunTime DATETIME2,
    NextRunTime DATETIME2,
    RunIntervalMinutes INT DEFAULT 60,
    Status NVARCHAR(20) DEFAULT 'Idle' -- Idle, Running, Failed, Disabled
);

-- Pre-populate discovery methods
INSERT INTO discovery.DiscoveryMethods (MethodName, MethodType, Description, Priority, RunIntervalMinutes)
VALUES
('SQL Server Browser', 'NetworkScan', 'Discover SQL instances via UDP 1434', 1, 60),
('Active Directory', 'APIConnector', 'Query AD for SQL Server SPNs', 2, 240),
('Azure SQL Discovery', 'APIConnector', 'Discover Azure SQL via ARM API', 3, 120),
('AWS RDS Discovery', 'APIConnector', 'Discover RDS instances via AWS API', 4, 120),
('VMware vCenter', 'APIConnector', 'Discover VMs running SQL Server', 5, 240),
('Central Management Server', 'SQLQuery', 'Query CMS for registered servers', 6, 30),
('DBAOps Agent', 'Agent', 'Local agent reporting', 7, 5),
('Manual Registration', 'ManualRegistration', 'Manually registered servers', 99, NULL),
('Kubernetes Discovery', 'APIConnector', 'Discover SQL in K8s namespaces', 8, 60),
('ServiceNow CMDB', 'APIConnector', 'Sync from ServiceNow CMDB', 9, 360);

-- Discovered Assets (raw discovery data)
CREATE TABLE discovery.DiscoveredAssets (
    DiscoveryID BIGINT IDENTITY(1,1) PRIMARY KEY,
    DiscoveredAt DATETIME2 DEFAULT GETDATE(),
    DiscoveryMethodID INT FOREIGN KEY REFERENCES discovery.DiscoveryMethods(MethodID),
    AssetType NVARCHAR(50), -- SQLServer, Database, VM, Container, Cluster
    
    -- Identifiers
    HostName NVARCHAR(256),
    FQDN NVARCHAR(256),
    IPAddress NVARCHAR(45),
    InstanceName NVARCHAR(128),
    Port INT,
    UUID NVARCHAR(100),
    
    -- Metadata
    OperatingSystem NVARCHAR(100),
    SQLVersion NVARCHAR(100),
    SQLEdition NVARCHAR(50),
    IsCluster BIT,
    IsClustered BIT,
    IsAGReplica BIT,
    AGName NVARCHAR(128),
    
    -- Cloud
    CloudProvider NVARCHAR(50), -- Azure, AWS, GCP, None
    SubscriptionID NVARCHAR(100),
    ResourceGroup NVARCHAR(100),
    Region NVARCHAR(50),
    
    -- Status
    ReconciliationStatus NVARCHAR(20) DEFAULT 'New', -- New, Matched, Merged, Ignored
    MatchedServerID INT,
    RawData NVARCHAR(MAX), -- Full JSON from discovery
    
    INDEX IX_Discovery_Host NONCLUSTERED (HostName, InstanceName),
    INDEX IX_Discovery_IP NONCLUSTERED (IPAddress)
);

-- Discovery Jobs
CREATE TABLE discovery.DiscoveryJobs (
    JobID INT IDENTITY(1,1) PRIMARY KEY,
    MethodID INT FOREIGN KEY REFERENCES discovery.DiscoveryMethods(MethodID),
    StartTime DATETIME2 DEFAULT GETDATE(),
    EndTime DATETIME2,
    Status NVARCHAR(20) DEFAULT 'Running', -- Running, Completed, Failed, Cancelled
    AssetsDiscovered INT DEFAULT 0,
    AssetsNew INT DEFAULT 0,
    AssetsUpdated INT DEFAULT 0,
    ErrorCount INT DEFAULT 0,
    ErrorDetails NVARCHAR(MAX),
    TriggeredBy NVARCHAR(128)
);

-- Agent Registration
CREATE TABLE discovery.AgentRegistry (
    AgentID UNIQUEIDENTIFIER PRIMARY KEY DEFAULT NEWID(),
    HostName NVARCHAR(256) NOT NULL,
    AgentVersion NVARCHAR(20),
    Platform NVARCHAR(50), -- Windows, Linux, Kubernetes
    Architecture NVARCHAR(20), -- x64, ARM64
    InstallPath NVARCHAR(500),
    InstalledAt DATETIME2,
    LastHeartbeat DATETIME2,
    HeartbeatIntervalSeconds INT DEFAULT 60,
    Status NVARCHAR(20) DEFAULT 'Active', -- Active, Inactive, Unresponsive, Uninstalling
    Capabilities NVARCHAR(MAX), -- JSON array of capabilities
    Configuration NVARCHAR(MAX), -- JSON config
    CertificateThumbprint NVARCHAR(100),
    CertificateExpiry DATETIME2,
    AutoUpdateEnabled BIT DEFAULT 1,
    UpdateChannel NVARCHAR(20) DEFAULT 'Stable' -- Stable, Preview, Beta
);

PRINT '✅ Discovery Mechanisms Schema Created';
