-- ============================================================
-- DBAOps: Network & Connectivity Readiness Schema
-- ============================================================
IF NOT EXISTS (SELECT 1 FROM sys.schemas WHERE name = 'network')
    EXEC('CREATE SCHEMA network');
GO

-- Required Ports/Endpoints
CREATE TABLE network.RequiredEndpoints (
    EndpointID INT IDENTITY(1,1) PRIMARY KEY,
    EndpointName NVARCHAR(100) NOT NULL,
    Category NVARCHAR(50), -- SQLServer, Azure, AWS, LDAP, API, Telemetry
    HostOrIP NVARCHAR(256),
    Port INT,
    Protocol NVARCHAR(10), -- TCP, UDP, HTTPS
    Direction NVARCHAR(10), -- Inbound, Outbound, Both
    IsRequired BIT DEFAULT 1,
    Description NVARCHAR(500),
    FirewallRule NVARCHAR(200),
    DocumentationURL NVARCHAR(500)
);

-- Pre-populate required endpoints
INSERT INTO network.RequiredEndpoints (EndpointName, Category, HostOrIP, Port, Protocol, Direction, Description)
VALUES
-- SQL Server
('SQL Server Default', 'SQLServer', '*', 1433, 'TCP', 'Outbound', 'SQL Server default instance'),
('SQL Server Browser', 'SQLServer', '*', 1434, 'UDP', 'Outbound', 'SQL Server Browser service'),
('SQL Server DAC', 'SQLServer', '*', 1434, 'TCP', 'Outbound', 'Dedicated Admin Connection'),
('SQL AG Endpoint', 'SQLServer', '*', 5022, 'TCP', 'Both', 'Availability Group endpoint'),

-- Azure
('Azure Management', 'Azure', 'management.azure.com', 443, 'HTTPS', 'Outbound', 'Azure Resource Manager'),
('Azure SQL', 'Azure', '*.database.windows.net', 1433, 'TCP', 'Outbound', 'Azure SQL Database'),
('Azure AD', 'Azure', 'login.microsoftonline.com', 443, 'HTTPS', 'Outbound', 'Azure AD authentication'),
('Azure Key Vault', 'Azure', '*.vault.azure.net', 443, 'HTTPS', 'Outbound', 'Azure Key Vault'),

-- AWS
('AWS RDS', 'AWS', '*.rds.amazonaws.com', 1433, 'TCP', 'Outbound', 'AWS RDS SQL Server'),
('AWS Secrets Manager', 'AWS', 'secretsmanager.*.amazonaws.com', 443, 'HTTPS', 'Outbound', 'AWS Secrets Manager'),

-- Directory Services
('LDAP', 'LDAP', '*', 389, 'TCP', 'Outbound', 'LDAP authentication'),
('LDAPS', 'LDAP', '*', 636, 'TCP', 'Outbound', 'LDAP over SSL'),
('Kerberos', 'LDAP', '*', 88, 'TCP', 'Outbound', 'Kerberos authentication'),
('DNS', 'DNS', '*', 53, 'UDP', 'Outbound', 'DNS resolution'),
('NTP', 'NTP', '*', 123, 'UDP', 'Outbound', 'Time synchronization');

-- Connectivity Test Results
CREATE TABLE network.ConnectivityTestResults (
    TestID BIGINT IDENTITY(1,1) PRIMARY KEY,
    TestTimestamp DATETIME2 DEFAULT GETDATE(),
    SourceHost NVARCHAR(256),
    EndpointID INT FOREIGN KEY REFERENCES network.RequiredEndpoints(EndpointID),
    TargetHost NVARCHAR(256),
    TargetPort INT,
    Protocol NVARCHAR(10),
    
    -- Results
    TestResult NVARCHAR(20), -- Success, Failed, Timeout, DNSFailed, TLSFailed, Blocked
    ResponseTimeMs INT,
    
    -- TLS Details
    TLSVersion NVARCHAR(20),
    CertificateSubject NVARCHAR(256),
    CertificateExpiry DATETIME2,
    CertificateIssuer NVARCHAR(256),
    CertificateChainValid BIT,
    
    -- DNS Details
    ResolvedIPAddress NVARCHAR(45),
    ReverseDNS NVARCHAR(256),
    
    -- NTP Details
    NTPOffset_ms INT,
    
    ErrorDetails NVARCHAR(MAX),
    Recommendation NVARCHAR(500),
    
    INDEX IX_ConnTest_Time NONCLUSTERED (TestTimestamp DESC)
);

-- Proxy Configuration
CREATE TABLE network.ProxyConfiguration (
    ConfigID INT IDENTITY(1,1) PRIMARY KEY,
    ProxyType NVARCHAR(20), -- HTTP, HTTPS, SOCKS5, PAC
    ProxyHost NVARCHAR(256),
    ProxyPort INT,
    PACFileURL NVARCHAR(500),
    BypassList NVARCHAR(MAX), -- Comma-separated hosts to bypass
    AuthRequired BIT DEFAULT 0,
    AuthCredentialID INT, -- FK to secrets
    IsDefault BIT DEFAULT 0,
    IsEnabled BIT DEFAULT 1,
    CreatedAt DATETIME2 DEFAULT GETDATE()
);

PRINT '✅ Network & Connectivity Schema Created';
