# ============================================================
# DBAOps Sentinel - Monitoring Agent
# Your Database Guardian - Always Watching
# ============================================================

$ErrorActionPreference = "Continue"
$configPath = $env:SENTINEL_CONFIG
if (-not $configPath) { $configPath = "C:\Program Files\DBAOps\Sentinel\Config\sentinel-config.json" }

# Load configuration
if (!(Test-Path $configPath)) {
    Write-EventLog -LogName Application -Source "DBAOps Sentinel" -EventId 1001 -EntryType Error -Message "Configuration file not found: $configPath"
    exit 1
}

$config = Get-Content $configPath | ConvertFrom-Json
$logPath = Join-Path $config.LogPath "sentinel-agent-$(Get-Date -Format 'yyyyMMdd').log"

function Write-Log {
    param([string]$Message, [string]$Level = "INFO")
    $timestamp = Get-Date -Format "yyyy-MM-dd HH:mm:ss"
    $logEntry = "[$timestamp] [$Level] $Message"
    Add-Content -Path $logPath -Value $logEntry
    if ($Level -eq "ERROR") {
        Write-EventLog -LogName Application -Source "DBAOps Sentinel" -EventId 1002 -EntryType Error -Message $Message -ErrorAction SilentlyContinue
    }
}

function Test-License {
    if ($config.License.Type -eq "Scout") {
        return @{ Valid = $true; MaxServers = 2; SelfHealing = $false }
    }
    
    if ($config.License.ExpiresAt) {
        $expires = [DateTime]::Parse($config.License.ExpiresAt)
        if ($expires -lt (Get-Date)) {
            Write-Log "License expired on $($config.License.ExpiresAt)" "WARNING"
            return @{ Valid = $false; MaxServers = 2; SelfHealing = $false }
        }
    }
    
    $limits = @{
        "Watch" = @{ MaxServers = 5; SelfHealing = $false }
        "Guard" = @{ MaxServers = 15; SelfHealing = $true }
        "Command" = @{ MaxServers = 50; SelfHealing = $true }
        "Fortress" = @{ MaxServers = 9999; SelfHealing = $true }
        "Trial" = @{ MaxServers = 15; SelfHealing = $true }
    }
    
    $tier = $limits[$config.License.Type]
    if ($tier) {
        return @{ Valid = $true; MaxServers = $tier.MaxServers; SelfHealing = $tier.SelfHealing }
    }
    
    return @{ Valid = $true; MaxServers = 2; SelfHealing = $false }
}

function Get-SQLServerMetrics {
    param([string]$ServerName, [string]$Database = "master")
    
    try {
        $connectionString = "Server=$ServerName;Database=$Database;Integrated Security=True;TrustServerCertificate=True"
        $connection = New-Object System.Data.SqlClient.SqlConnection($connectionString)
        $connection.Open()
        
        $metrics = @{
            ServerName = $ServerName
            Timestamp = Get-Date
            IsOnline = $true
        }
        
        # Get CPU usage
        $cpuQuery = @"
SELECT 
    record.value('(./Record/SchedulerMonitorEvent/SystemHealth/ProcessUtilization)[1]', 'int') AS CPU
FROM (
    SELECT TOP 1 CONVERT(XML, record) AS record
    FROM sys.dm_os_ring_buffers
    WHERE ring_buffer_type = N'RING_BUFFER_SCHEDULER_MONITOR'
    ORDER BY timestamp DESC
) AS x
"@
        $cmd = New-Object System.Data.SqlClient.SqlCommand($cpuQuery, $connection)
        $metrics.CPUPercent = $cmd.ExecuteScalar()
        
        # Get memory usage
        $memQuery = "SELECT (physical_memory_in_use_kb * 100.0 / total_physical_memory_kb) FROM sys.dm_os_process_memory"
        $cmd = New-Object System.Data.SqlClient.SqlCommand($memQuery, $connection)
        $metrics.MemoryPercent = [math]::Round($cmd.ExecuteScalar(), 2)
        
        # Get connection count
        $connQuery = "SELECT COUNT(*) FROM sys.dm_exec_sessions WHERE is_user_process = 1"
        $cmd = New-Object System.Data.SqlClient.SqlCommand($connQuery, $connection)
        $metrics.ConnectionCount = $cmd.ExecuteScalar()
        
        $connection.Close()
        return $metrics
        
    } catch {
        Write-Log "Failed to collect metrics from $ServerName : $_" "ERROR"
        return @{
            ServerName = $ServerName
            Timestamp = Get-Date
            IsOnline = $false
            Error = $_.Exception.Message
        }
    }
}

function Save-Metrics {
    param($Metrics)
    
    $dataFile = Join-Path $config.DataPath "metrics-$(Get-Date -Format 'yyyyMMdd').json"
    
    if (Test-Path $dataFile) {
        $existing = Get-Content $dataFile | ConvertFrom-Json
        $existing += $Metrics
        $existing | ConvertTo-Json -Depth 10 | Out-File $dataFile -Encoding UTF8 -Force
    } else {
        @($Metrics) | ConvertTo-Json -Depth 10 | Out-File $dataFile -Encoding UTF8 -Force
    }
}

# Main agent loop
Write-Log "🛡️ DBAOps Sentinel Agent started"
Write-Log "Configuration: $configPath"

# Register event source
New-EventLog -LogName Application -Source "DBAOps Sentinel" -ErrorAction SilentlyContinue

$license = Test-License
Write-Log "License: $($config.License.Type) | Max Servers: $($license.MaxServers) | Self-Healing: $($license.SelfHealing)"

# Get monitored servers from repository or config
$servers = @($config.Database.Server)

while ($true) {
    try {
        foreach ($server in $servers | Select-Object -First $license.MaxServers) {
            Write-Log "Collecting metrics from $server"
            $metrics = Get-SQLServerMetrics -ServerName $server
            Save-Metrics -Metrics $metrics
            
            # Check thresholds and alert
            if ($metrics.CPUPercent -gt 85) {
                Write-Log "HIGH CPU on $server : $($metrics.CPUPercent)%" "WARNING"
            }
            if ($metrics.MemoryPercent -gt 90) {
                Write-Log "HIGH MEMORY on $server : $($metrics.MemoryPercent)%" "WARNING"
            }
        }
        
        Write-Log "Collection cycle complete. Sleeping for $($config.Monitoring.IntervalSeconds) seconds"
        Start-Sleep -Seconds $config.Monitoring.IntervalSeconds
        
    } catch {
        Write-Log "Agent error: $_" "ERROR"
        Start-Sleep -Seconds 60
    }
}
