// ============================================================
// DBAOps Sentinel - License Validation API
// Add this to your existing server.js
// ============================================================

// LICENSE ENDPOINTS
// -----------------

// Activate License
app.post('/api/license/activate', async (req, res) => {
    try {
        const { key, machineId, hostname, ip } = req.body;
        
        if (!key || !machineId) {
            return res.status(400).json({ valid: false, message: 'License key and machine ID required' });
        }
        
        const pool = await getPool();
        const result = await pool.request()
            .input('LicenseKey', sql.NVarChar, key)
            .input('MachineID', sql.NVarChar, machineId)
            .input('MachineName', sql.NVarChar, hostname || 'Unknown')
            .input('MachineIP', sql.NVarChar, ip || req.ip)
            .output('Success', sql.Bit)
            .output('Message', sql.NVarChar(500))
            .output('TierCode', sql.NVarChar(20))
            .output('ExpiresAt', sql.DateTime2)
            .execute('licensing.ActivateLicense');
        
        const success = result.output.Success;
        
        res.json({
            valid: success,
            message: result.output.Message,
            tier: result.output.TierCode,
            expiresAt: result.output.ExpiresAt
        });
    } catch (err) {
        console.error('License activation error:', err);
        res.status(500).json({ valid: false, message: 'License server error' });
    }
});

// Check License Status
app.get('/api/license/status/:key', async (req, res) => {
    try {
        const pool = await getPool();
        const result = await pool.request()
            .input('LicenseKey', sql.NVarChar, req.params.key)
            .execute('licensing.CheckLicenseStatus');
        
        if (result.recordsets[0].length === 0) {
            return res.status(404).json({ valid: false, message: 'License not found' });
        }
        
        const license = result.recordsets[0][0];
        const activations = result.recordsets[1] || [];
        
        res.json({
            valid: license.IsValid === 1,
            key: license.LicenseKey,
            tier: license.TierCode,
            tierName: license.TierName,
            customer: license.CustomerName,
            status: license.Status,
            maxServers: license.MaxServers,
            maxDatabases: license.MaxDatabases,
            expiresAt: license.ExpiresAt,
            daysRemaining: license.DaysRemaining,
            activations: {
                current: license.ActivationCount,
                max: license.MaxActivations,
                machines: activations
            }
        });
    } catch (err) {
        res.status(500).json({ error: err.message });
    }
});

// Deactivate License
app.post('/api/license/deactivate', async (req, res) => {
    try {
        const { key, machineId } = req.body;
        
        const pool = await getPool();
        await pool.request()
            .input('LicenseKey', sql.NVarChar, key)
            .input('MachineID', sql.NVarChar, machineId)
            .execute('licensing.DeactivateLicense');
        
        res.json({ success: true, message: 'License deactivated' });
    } catch (err) {
        res.status(500).json({ error: err.message });
    }
});

// Generate License (Admin only - add authentication)
app.post('/api/license/generate', async (req, res) => {
    try {
        const { tier, customerName, customerEmail, validDays } = req.body;
        
        const pool = await getPool();
        const result = await pool.request()
            .input('TierCode', sql.NVarChar, tier)
            .input('CustomerName', sql.NVarChar, customerName)
            .input('CustomerEmail', sql.NVarChar, customerEmail)
            .input('ValidDays', sql.Int, validDays || 365)
            .input('CreatedBy', sql.NVarChar, 'API')
            .output('LicenseKey', sql.NVarChar(30))
            .execute('licensing.GenerateLicenseKey');
        
        res.json({
            success: true,
            licenseKey: result.output.LicenseKey,
            tier,
            customer: customerName,
            expiresAt: new Date(Date.now() + (validDays || 365) * 24 * 60 * 60 * 1000)
        });
    } catch (err) {
        res.status(500).json({ error: err.message });
    }
});

// Heartbeat (for usage tracking)
app.post('/api/license/heartbeat', async (req, res) => {
    try {
        const { key, machineId, serverCount, databaseCount } = req.body;
        
        const pool = await getPool();
        await pool.request()
            .input('LicenseKey', sql.NVarChar, key)
            .input('MachineID', sql.NVarChar, machineId)
            .input('ServerCount', sql.Int, serverCount || 0)
            .input('DatabaseCount', sql.Int, databaseCount || 0)
            .query(`
                UPDATE licensing.LicenseActivations 
                SET LastHeartbeat = GETDATE(), ServerCount = @ServerCount, DatabaseCount = @DatabaseCount
                WHERE LicenseKey = @LicenseKey AND MachineID = @MachineID AND IsActive = 1
            `);
        
        res.json({ success: true });
    } catch (err) {
        res.status(500).json({ error: err.message });
    }
});
