-- ============================================================
-- DBAOps: Inventory Model & Collectors Schema
-- ============================================================

-- Canonical Asset Schema (Capability 5)
IF NOT EXISTS (SELECT 1 FROM sys.schemas WHERE name = 'inventory')
    EXEC('CREATE SCHEMA inventory');
GO

-- Master Asset Table
CREATE TABLE inventory.Assets (
    AssetID INT IDENTITY(1,1) PRIMARY KEY,
    AssetGUID UNIQUEIDENTIFIER DEFAULT NEWID(),
    AssetType NVARCHAR(50) NOT NULL, -- Host, VM, SQLInstance, Database, Cluster, AG, Container
    
    -- Primary Identifiers
    Name NVARCHAR(256) NOT NULL,
    FQDN NVARCHAR(256),
    IPAddresses NVARCHAR(MAX), -- JSON array
    UUID NVARCHAR(100),
    
    -- Hierarchy
    ParentAssetID INT FOREIGN KEY REFERENCES inventory.Assets(AssetID),
    
    -- SQL Specific
    SQLInstanceName NVARCHAR(128),
    SQLVersion NVARCHAR(50),
    SQLEdition NVARCHAR(50),
    SQLProductLevel NVARCHAR(20),
    SQLBuild NVARCHAR(50),
    Port INT,
    
    -- Environment
    Environment NVARCHAR(50), -- Production, Development, Test, DR, etc.
    Criticality NVARCHAR(20), -- Critical, High, Medium, Low
    Tier NVARCHAR(20), -- Tier1, Tier2, Tier3
    
    -- Ownership
    Owner NVARCHAR(256),
    OwnerEmail NVARCHAR(256),
    Team NVARCHAR(100),
    CostCenter NVARCHAR(50),
    Department NVARCHAR(100),
    
    -- Cloud
    CloudProvider NVARCHAR(50),
    SubscriptionID NVARCHAR(100),
    ResourceGroup NVARCHAR(100),
    Region NVARCHAR(50),
    ResourceID NVARCHAR(500),
    
    -- Tags
    Tags NVARCHAR(MAX), -- JSON key-value pairs
    
    -- Source of Truth
    SourcePriority INT DEFAULT 100,
    PrimarySource NVARCHAR(50), -- CMDB, AD, Discovery, Manual
    LastSourceUpdate DATETIME2,
    
    -- Status
    Status NVARCHAR(20) DEFAULT 'Active', -- Active, Inactive, Decommissioned, Unknown
    IsManaged BIT DEFAULT 1,
    IsSuppressed BIT DEFAULT 0,
    
    -- Timestamps
    DiscoveredAt DATETIME2,
    CreatedAt DATETIME2 DEFAULT GETDATE(),
    ModifiedAt DATETIME2 DEFAULT GETDATE(),
    ModifiedBy NVARCHAR(128),
    
    -- Reconciliation
    CMDBAssetID NVARCHAR(100),
    ServiceNowSysID NVARCHAR(50),
    ADObjectGUID NVARCHAR(100),
    
    INDEX IX_Asset_Name NONCLUSTERED (Name),
    INDEX IX_Asset_FQDN NONCLUSTERED (FQDN),
    INDEX IX_Asset_Type NONCLUSTERED (AssetType, Status)
);

-- Asset Change History
CREATE TABLE inventory.AssetChangeHistory (
    ChangeID BIGINT IDENTITY(1,1) PRIMARY KEY,
    AssetID INT FOREIGN KEY REFERENCES inventory.Assets(AssetID),
    ChangeTimestamp DATETIME2 DEFAULT GETDATE(),
    ChangeType NVARCHAR(20), -- Created, Updated, Deleted, Merged
    FieldName NVARCHAR(100),
    OldValue NVARCHAR(MAX),
    NewValue NVARCHAR(MAX),
    ChangeSource NVARCHAR(50), -- Discovery, Manual, API, Sync
    ChangedBy NVARCHAR(128)
);

-- Source Priority Rules
CREATE TABLE inventory.SourcePriorityRules (
    RuleID INT IDENTITY(1,1) PRIMARY KEY,
    SourceName NVARCHAR(50) NOT NULL,
    Priority INT NOT NULL, -- Lower = higher priority
    FieldsControlled NVARCHAR(MAX), -- JSON array of fields this source controls
    IsActive BIT DEFAULT 1
);

INSERT INTO inventory.SourcePriorityRules (SourceName, Priority, FieldsControlled)
VALUES
('ServiceNow CMDB', 1, '["Owner","Environment","Criticality","CostCenter"]'),
('Active Directory', 2, '["FQDN","OperatingSystem"]'),
('DBAOps Discovery', 3, '["SQLVersion","SQLEdition","Port","IPAddresses"]'),
('Manual Entry', 4, '["Tags","Notes"]');

-- Collectors (Capability 6)
IF NOT EXISTS (SELECT 1 FROM sys.schemas WHERE name = 'collector')
    EXEC('CREATE SCHEMA collector');
GO

CREATE TABLE collector.CollectorDefinitions (
    CollectorID INT IDENTITY(1,1) PRIMARY KEY,
    CollectorName NVARCHAR(100) NOT NULL,
    CollectorType NVARCHAR(50) NOT NULL, -- Polling, EventDriven, Webhook, Scheduled
    Description NVARCHAR(500),
    TargetAssetType NVARCHAR(50),
    
    -- Scheduling
    IsEnabled BIT DEFAULT 1,
    ScheduleType NVARCHAR(20), -- Interval, Cron, EventDriven
    IntervalMinutes INT,
    CronExpression NVARCHAR(100),
    
    -- Execution
    ExecutionScript NVARCHAR(MAX),
    TimeoutSeconds INT DEFAULT 300,
    RetryCount INT DEFAULT 3,
    RetryDelaySeconds INT DEFAULT 30,
    BackoffMultiplier DECIMAL(3,1) DEFAULT 2.0,
    
    -- Throttling
    MaxConcurrent INT DEFAULT 5,
    ThrottlePerMinute INT DEFAULT 60,
    
    -- Status
    LastRunTime DATETIME2,
    NextRunTime DATETIME2,
    LastRunStatus NVARCHAR(20),
    ConsecutiveFailures INT DEFAULT 0,
    
    CreatedAt DATETIME2 DEFAULT GETDATE()
);

-- Pre-populate collectors
INSERT INTO collector.CollectorDefinitions 
(CollectorName, CollectorType, Description, TargetAssetType, ScheduleType, IntervalMinutes)
VALUES
('Server Health', 'Polling', 'Collect CPU, Memory, Disk metrics', 'SQLInstance', 'Interval', 1),
('Database Status', 'Polling', 'Collect database state and sizes', 'Database', 'Interval', 5),
('Backup Status', 'Polling', 'Check backup history', 'Database', 'Interval', 15),
('Job Status', 'Polling', 'SQL Agent job status', 'SQLInstance', 'Interval', 5),
('Wait Stats', 'Polling', 'Collect wait statistics', 'SQLInstance', 'Interval', 1),
('AG Health', 'Polling', 'Availability Group status', 'SQLInstance', 'Interval', 1),
('Security Events', 'EventDriven', 'Login failures, permission changes', 'SQLInstance', 'EventDriven', NULL),
('Configuration Changes', 'EventDriven', 'sp_configure changes', 'SQLInstance', 'EventDriven', NULL);

-- Collector Execution Log
CREATE TABLE collector.ExecutionLog (
    ExecutionID BIGINT IDENTITY(1,1) PRIMARY KEY,
    CollectorID INT FOREIGN KEY REFERENCES collector.CollectorDefinitions(CollectorID),
    StartTime DATETIME2 DEFAULT GETDATE(),
    EndTime DATETIME2,
    Duration_ms INT,
    Status NVARCHAR(20), -- Running, Success, Failed, Timeout, Throttled
    AssetsProcessed INT DEFAULT 0,
    DataPointsCollected INT DEFAULT 0,
    ErrorCount INT DEFAULT 0,
    ErrorDetails NVARCHAR(MAX),
    INDEX IX_Execution_Time NONCLUSTERED (StartTime DESC)
);

PRINT '✅ Inventory Model & Collectors Schema Created';
