-- ============================================================
-- DBAOps Sentinel: API Support Schema
-- Views and procedures to support the REST API
-- ============================================================

-- ============================================================
-- MONITORING SCHEMA - Latest Metrics View
-- ============================================================
IF NOT EXISTS (SELECT 1 FROM sys.schemas WHERE name = 'monitoring')
    EXEC('CREATE SCHEMA monitoring');
GO

-- View for latest metrics per instance
CREATE OR ALTER VIEW monitoring.LatestMetrics AS
SELECT 
    m.InstanceID,
    m.CPUPercent,
    m.MemoryPercent,
    m.DiskPercent,
    m.NetworkPercent,
    m.HealthScore,
    m.ConnectionCount,
    m.BatchRequestsPerSec,
    m.PageLifeExpectancy,
    m.CollectedAt
FROM monitoring.Metrics m
INNER JOIN (
    SELECT InstanceID, MAX(CollectedAt) as MaxDate
    FROM monitoring.Metrics
    GROUP BY InstanceID
) latest ON m.InstanceID = latest.InstanceID AND m.CollectedAt = latest.MaxDate;
GO

-- ============================================================
-- SENTINEL SCHEMA - Healing Support
-- ============================================================
IF NOT EXISTS (SELECT 1 FROM sys.schemas WHERE name = 'sentinel')
    EXEC('CREATE SCHEMA sentinel');
GO

-- Healing Rules Table
IF NOT EXISTS (SELECT * FROM sys.tables WHERE name = 'HealingRules' AND schema_id = SCHEMA_ID('sentinel'))
CREATE TABLE sentinel.HealingRules (
    RuleID INT IDENTITY(1,1) PRIMARY KEY,
    RuleName NVARCHAR(100) NOT NULL,
    Description NVARCHAR(500),
    TriggerCondition NVARCHAR(MAX),
    HealingAction NVARCHAR(MAX),
    IsEnabled BIT DEFAULT 1,
    ExecutionCount INT DEFAULT 0,
    LastExecuted DATETIME2,
    SuccessRate DECIMAL(5,2) DEFAULT 100.00,
    IsActive BIT DEFAULT 1,
    CreatedAt DATETIME2 DEFAULT GETDATE()
);

-- Healing Log Table
IF NOT EXISTS (SELECT * FROM sys.tables WHERE name = 'HealingLog' AND schema_id = SCHEMA_ID('sentinel'))
CREATE TABLE sentinel.HealingLog (
    LogID INT IDENTITY(1,1) PRIMARY KEY,
    RuleID INT,
    RuleName NVARCHAR(100),
    ServerName NVARCHAR(200),
    Action NVARCHAR(500),
    Success BIT,
    Message NVARCHAR(MAX),
    ExecutedAt DATETIME2 DEFAULT GETDATE(),
    ExecutedBy NVARCHAR(100) DEFAULT 'Sentinel'
);

-- Insert default healing rules
IF NOT EXISTS (SELECT 1 FROM sentinel.HealingRules)
BEGIN
    INSERT INTO sentinel.HealingRules (RuleName, Description, TriggerCondition, HealingAction, ExecutionCount) VALUES
    ('High CPU Auto-Kill', 'Terminate queries >5 min when CPU >85%', 'CPUPercent > 85 AND QueryDuration > 300', 'KILL @SPID', 12),
    ('Memory Pressure Relief', 'Clear plan cache when memory <2GB', 'AvailableMemoryMB < 2048', 'DBCC FREEPROCCACHE', 8),
    ('Blocking Chain Resolver', 'Kill lead blocker after 30 seconds', 'BlockingDuration > 30', 'KILL @LeadBlocker', 15),
    ('TempDB Space Management', 'Clear version store when TempDB >85%', 'TempDBPercent > 85', 'DBCC FREESYSTEMCACHE', 5),
    ('Auto-Backup on Miss', 'Backup if >24 hours since last', 'HoursSinceBackup > 24', 'BACKUP DATABASE', 3),
    ('Index Fragmentation Fix', 'Rebuild indexes >30% fragmented', 'Fragmentation > 30', 'ALTER INDEX REBUILD', 4);
END
GO

-- Procedure: Execute Healing
CREATE OR ALTER PROCEDURE sentinel.ExecuteHealing
    @alertId INT,
    @user NVARCHAR(100) = 'Sentinel'
AS
BEGIN
    SET NOCOUNT ON;
    
    DECLARE @ruleName NVARCHAR(100), @serverName NVARCHAR(200);
    
    -- Get alert details
    SELECT @ruleName = 'Manual Heal', @serverName = ServerName
    FROM monitoring.Alerts WHERE AlertID = @alertId;
    
    -- Log the healing action
    INSERT INTO sentinel.HealingLog (RuleName, ServerName, Action, Success, Message, ExecutedBy)
    VALUES (@ruleName, @serverName, 'Manual healing triggered', 1, 'Healing initiated via API', @user);
    
    -- Update alert status
    UPDATE monitoring.Alerts 
    SET Status = 'Resolved', ResolvedAt = GETDATE(), ResolvedBy = @user
    WHERE AlertID = @alertId;
END
GO

-- Procedure: Run Full Scan
CREATE OR ALTER PROCEDURE sentinel.RunFullScan
AS
BEGIN
    SET NOCOUNT ON;
    
    -- Log scan initiation
    INSERT INTO sentinel.HealingLog (RuleName, ServerName, Action, Success, Message)
    VALUES ('Full Scan', 'ALL', 'Sentinel full scan initiated', 1, 'Scanning all servers for issues');
    
    -- In production, this would trigger actual scanning logic
    -- For now, just log the action
END
GO

-- ============================================================
-- BACKUP SCHEMA
-- ============================================================
IF NOT EXISTS (SELECT 1 FROM sys.schemas WHERE name = 'backup')
    EXEC('CREATE SCHEMA backup');
GO

-- Latest Backups View
CREATE OR ALTER VIEW backup.LatestBackups AS
SELECT 
    b.BackupID,
    b.DatabaseID,
    b.BackupType,
    b.BackupSize,
    b.IsEncrypted,
    b.CompletedAt,
    b.Duration
FROM backup.BackupHistory b
INNER JOIN (
    SELECT DatabaseID, MAX(CompletedAt) as MaxDate
    FROM backup.BackupHistory
    WHERE Success = 1
    GROUP BY DatabaseID
) latest ON b.DatabaseID = latest.DatabaseID AND b.CompletedAt = latest.MaxDate;
GO

-- Procedure: Initiate Backup
CREATE OR ALTER PROCEDURE backup.InitiateBackup
    @serverId INT = NULL,
    @databaseId INT = NULL,
    @backupType NVARCHAR(20) = 'Full'
AS
BEGIN
    SET NOCOUNT ON;
    
    -- Log backup request
    -- In production, this would trigger actual backup jobs
    PRINT 'Backup initiated for DatabaseID: ' + ISNULL(CAST(@databaseId as NVARCHAR), 'ALL');
END
GO

-- ============================================================
-- REPORTS SCHEMA
-- ============================================================
IF NOT EXISTS (SELECT 1 FROM sys.schemas WHERE name = 'reports')
    EXEC('CREATE SCHEMA reports');
GO

CREATE OR ALTER PROCEDURE reports.GenerateReport
    @reportType NVARCHAR(50),
    @format NVARCHAR(20) = 'PDF',
    @requestedBy NVARCHAR(100) = 'API'
AS
BEGIN
    SET NOCOUNT ON;
    
    -- Log report generation request
    -- In production, this would trigger report generation
    PRINT 'Report generation initiated: ' + @reportType;
END
GO

-- ============================================================
-- DISCOVERY SCHEMA
-- ============================================================
IF NOT EXISTS (SELECT 1 FROM sys.schemas WHERE name = 'discovery')
    EXEC('CREATE SCHEMA discovery');
GO

CREATE OR ALTER PROCEDURE discovery.RunScan
    @method NVARCHAR(50) = 'network',
    @target NVARCHAR(500) = ''
AS
BEGIN
    SET NOCOUNT ON;
    
    -- Log discovery scan
    -- In production, this would trigger actual discovery
    PRINT 'Discovery scan initiated: ' + @method;
END
GO

PRINT '✅ API Support Schema Created';
