-- ============================================================
-- DBAOps Ultimate: Comprehensive Audit & Security Schema
-- Industry-Compliant Retention & Critical Alert Framework
-- ============================================================

-- ============================================================
-- AUDIT SCHEMA - Complete Activity Tracking
-- ============================================================
IF NOT EXISTS (SELECT 1 FROM sys.schemas WHERE name = 'audit')
    EXEC('CREATE SCHEMA audit');
GO

-- Master Audit Log - All Activities
CREATE TABLE audit.ActivityLog (
    LogID BIGINT IDENTITY(1,1) PRIMARY KEY,
    ActivityTimestamp DATETIME2 DEFAULT GETDATE(),
    ServerName NVARCHAR(128),
    DatabaseName NVARCHAR(128),
    ObjectName NVARCHAR(256),
    ActivityType NVARCHAR(50) NOT NULL, -- Login, Query, DDL, DML, Admin, Security, Backup, Config
    ActivitySubType NVARCHAR(100),
    PrincipalName NVARCHAR(128),
    PrincipalType NVARCHAR(50), -- SQLLogin, WindowsLogin, ApplicationRole, gMSA
    ClientHostName NVARCHAR(128),
    ClientIPAddress NVARCHAR(45),
    ApplicationName NVARCHAR(256),
    SessionID INT,
    RequestID INT,
    StatementText NVARCHAR(MAX),
    ObjectsAccessed NVARCHAR(MAX), -- JSON array of objects
    RowsAffected BIGINT,
    Duration_ms BIGINT,
    Result NVARCHAR(20), -- Success, Failed, Denied, Error
    ErrorNumber INT,
    ErrorMessage NVARCHAR(MAX),
    RiskLevel NVARCHAR(20), -- Critical, High, Medium, Low, Info
    ComplianceFlags NVARCHAR(500), -- CJIS, HIPAA, PCI, SOX, etc.
    DataClassification NVARCHAR(50), -- PII, PHI, PCI, CJI, Confidential, Public
    RetentionCategory NVARCHAR(50),
    IsAlerted BIT DEFAULT 0,
    INDEX IX_ActivityLog_Time NONCLUSTERED (ActivityTimestamp),
    INDEX IX_ActivityLog_Server NONCLUSTERED (ServerName, DatabaseName),
    INDEX IX_ActivityLog_Principal NONCLUSTERED (PrincipalName),
    INDEX IX_ActivityLog_Risk NONCLUSTERED (RiskLevel)
) WITH (DATA_COMPRESSION = PAGE);

-- Data Access Audit (Sensitive Data Tracking)
CREATE TABLE audit.DataAccessLog (
    AccessID BIGINT IDENTITY(1,1) PRIMARY KEY,
    AccessTimestamp DATETIME2 DEFAULT GETDATE(),
    ServerName NVARCHAR(128),
    DatabaseName NVARCHAR(128),
    SchemaName NVARCHAR(128),
    TableName NVARCHAR(128),
    ColumnName NVARCHAR(128),
    DataClassification NVARCHAR(50), -- PII, PHI, PCI, CJI, SSN, DOB, Financial
    AccessType NVARCHAR(20), -- SELECT, INSERT, UPDATE, DELETE, EXPORT
    PrincipalName NVARCHAR(128),
    ClientIPAddress NVARCHAR(45),
    ApplicationName NVARCHAR(256),
    RowsAccessed BIGINT,
    QueryHash VARBINARY(64),
    WasEncrypted BIT,
    WasMasked BIT,
    ComplianceFramework NVARCHAR(100),
    RetentionDays INT,
    INDEX IX_DataAccess_Time NONCLUSTERED (AccessTimestamp),
    INDEX IX_DataAccess_Classification NONCLUSTERED (DataClassification)
);

-- Security Event Log (Authentication, Authorization, Threats)
CREATE TABLE audit.SecurityEventLog (
    EventID BIGINT IDENTITY(1,1) PRIMARY KEY,
    EventTimestamp DATETIME2 DEFAULT GETDATE(),
    EventCategory NVARCHAR(50), -- Authentication, Authorization, ThreatDetection, PolicyViolation
    EventType NVARCHAR(100),
    Severity NVARCHAR(20), -- Critical, High, Medium, Low
    ServerName NVARCHAR(128),
    PrincipalName NVARCHAR(128),
    SourceIPAddress NVARCHAR(45),
    TargetObject NVARCHAR(256),
    EventDescription NVARCHAR(MAX),
    ThreatIndicators NVARCHAR(MAX), -- JSON
    MITREAttackTactic NVARCHAR(100),
    MITREAttackTechnique NVARCHAR(100),
    ActionTaken NVARCHAR(100),
    IsResolved BIT DEFAULT 0,
    ResolvedBy NVARCHAR(128),
    ResolvedAt DATETIME2,
    EscalationLevel INT DEFAULT 0,
    INDEX IX_Security_Time NONCLUSTERED (EventTimestamp),
    INDEX IX_Security_Severity NONCLUSTERED (Severity)
);

-- Configuration Change Audit
CREATE TABLE audit.ConfigurationChangeLog (
    ChangeID BIGINT IDENTITY(1,1) PRIMARY KEY,
    ChangeTimestamp DATETIME2 DEFAULT GETDATE(),
    ServerName NVARCHAR(128),
    DatabaseName NVARCHAR(128),
    ConfigurationArea NVARCHAR(100), -- ServerConfig, DatabaseConfig, Security, Jobs, Replication
    SettingName NVARCHAR(256),
    OldValue NVARCHAR(MAX),
    NewValue NVARCHAR(MAX),
    ChangedBy NVARCHAR(128),
    ChangeMethod NVARCHAR(50), -- SSMS, Script, API, DBAOps
    ChangeReason NVARCHAR(500),
    ChangeTicket NVARCHAR(50),
    ApprovedBy NVARCHAR(128),
    RollbackScript NVARCHAR(MAX),
    ComplianceImpact NVARCHAR(500),
    INDEX IX_Config_Time NONCLUSTERED (ChangeTimestamp)
);

-- ============================================================
-- RETENTION POLICIES - Compliance-Based
-- ============================================================
CREATE TABLE audit.RetentionPolicies (
    PolicyID INT IDENTITY(1,1) PRIMARY KEY,
    PolicyName NVARCHAR(100) NOT NULL,
    Description NVARCHAR(500),
    ComplianceFramework NVARCHAR(100), -- CJIS, HIPAA, PCI-DSS, SOX, GDPR, FERPA
    DataCategory NVARCHAR(100),
    RetentionDays INT NOT NULL,
    RetentionYears AS (RetentionDays / 365),
    ArchiveAfterDays INT,
    PurgeAfterDays INT,
    LegalHoldOverride BIT DEFAULT 0,
    IsActive BIT DEFAULT 1,
    ApplicableIndustries NVARCHAR(500),
    RegulatoryReference NVARCHAR(500),
    CreatedDate DATETIME2 DEFAULT GETDATE(),
    LastModified DATETIME2
);

-- Pre-populate Retention Policies
INSERT INTO audit.RetentionPolicies (PolicyName, ComplianceFramework, DataCategory, RetentionDays, ArchiveAfterDays, ApplicableIndustries, RegulatoryReference)
VALUES
-- CJIS (Law Enforcement)
('CJIS Audit Logs', 'CJIS', 'SecurityAudit', 365, 180, 'Law Enforcement,Government', 'CJIS Security Policy 5.4.1.1'),
('CJIS Access Logs', 'CJIS', 'DataAccess', 365, 180, 'Law Enforcement,Government', 'CJIS Security Policy 5.4.1.1'),
('CJIS Transaction Logs', 'CJIS', 'Transaction', 365, 180, 'Law Enforcement,Government', 'CJIS Security Policy 5.4.1.1'),

-- HIPAA (Healthcare)
('HIPAA Audit Trail', 'HIPAA', 'PHIAccess', 2190, 365, 'Healthcare,Insurance', 'HIPAA 45 CFR 164.312(b)'),
('HIPAA Security Logs', 'HIPAA', 'SecurityAudit', 2190, 365, 'Healthcare,Insurance', 'HIPAA 45 CFR 164.312(b)'),
('HIPAA Transaction Records', 'HIPAA', 'Transaction', 2190, 365, 'Healthcare,Insurance', '6 years from creation'),

-- PCI-DSS (Banking/Finance)
('PCI Audit Trail', 'PCI-DSS', 'CardholderAccess', 365, 90, 'Banking,Finance,Retail', 'PCI-DSS Requirement 10.7'),
('PCI Security Events', 'PCI-DSS', 'SecurityAudit', 365, 90, 'Banking,Finance,Retail', 'PCI-DSS Requirement 10.7'),
('PCI Daily Logs', 'PCI-DSS', 'DailyOperations', 90, 30, 'Banking,Finance,Retail', 'PCI-DSS Requirement 10.7'),

-- SOX (Public Companies)
('SOX Financial Audit', 'SOX', 'FinancialData', 2555, 365, 'Public Companies,Finance', 'SOX Section 802'),
('SOX Access Logs', 'SOX', 'DataAccess', 2555, 365, 'Public Companies,Finance', '7 years retention'),

-- GDPR (EU Data)
('GDPR Processing Records', 'GDPR', 'PersonalData', 1095, 365, 'EU Operations,International', 'GDPR Article 30'),
('GDPR Consent Records', 'GDPR', 'Consent', 1825, 365, 'EU Operations,International', '5 years after relationship'),

-- FERPA (Education)
('FERPA Student Records', 'FERPA', 'StudentData', 1825, 365, 'Education,Academic', 'FERPA 34 CFR 99.10'),

-- General
('Standard Audit Logs', 'General', 'AuditTrail', 365, 180, 'All', 'Best Practice'),
('Performance Metrics', 'General', 'Performance', 90, 30, 'All', 'Operational');

-- ============================================================
-- CRITICAL ALERT DEFINITIONS
-- ============================================================
IF NOT EXISTS (SELECT 1 FROM sys.schemas WHERE name = 'alerts')
    EXEC('CREATE SCHEMA alerts');
GO

CREATE TABLE alerts.CriticalAlertDefinitions (
    AlertDefID INT IDENTITY(1,1) PRIMARY KEY,
    AlertCode NVARCHAR(20) NOT NULL UNIQUE,
    AlertName NVARCHAR(100) NOT NULL,
    AlertCategory NVARCHAR(50) NOT NULL, -- Security, Availability, Performance, Compliance, DataProtection
    Severity NVARCHAR(20) NOT NULL, -- Critical, High, Medium, Low
    Description NVARCHAR(500),
    DetectionQuery NVARCHAR(MAX),
    TriggerCondition NVARCHAR(500),
    ResponseTimeMinutes INT, -- Required response time
    EscalationMinutes INT, -- Time before escalation
    NotificationChannels NVARCHAR(500), -- Email, SMS, PagerDuty, Teams, Slack, Phone
    RequiresAcknowledgment BIT DEFAULT 1,
    AutoRemediation BIT DEFAULT 0,
    RemediationScript NVARCHAR(MAX),
    RunbookID INT,
    ComplianceFrameworks NVARCHAR(500),
    ApplicableIndustries NVARCHAR(500),
    IsActive BIT DEFAULT 1
);

-- Pre-populate Critical Alert Definitions
INSERT INTO alerts.CriticalAlertDefinitions 
(AlertCode, AlertName, AlertCategory, Severity, Description, ResponseTimeMinutes, EscalationMinutes, NotificationChannels, ComplianceFrameworks, ApplicableIndustries)
VALUES
-- SECURITY ALERTS (Critical)
('SEC-001', 'Data Breach Detected', 'Security', 'Critical', 'Potential unauthorized data exfiltration or access detected', 5, 15, 'Phone,SMS,Email,PagerDuty', 'CJIS,HIPAA,PCI-DSS,GDPR', 'All'),
('SEC-002', 'SQL Injection Attempt', 'Security', 'Critical', 'SQL injection attack pattern detected in query', 5, 15, 'Phone,SMS,Email,PagerDuty', 'All', 'All'),
('SEC-003', 'Brute Force Attack', 'Security', 'Critical', 'Multiple failed login attempts from same source', 10, 30, 'SMS,Email,PagerDuty', 'CJIS,PCI-DSS', 'All'),
('SEC-004', 'Privilege Escalation', 'Security', 'Critical', 'Unauthorized elevation of privileges detected', 5, 15, 'Phone,SMS,Email', 'CJIS,HIPAA,PCI-DSS', 'All'),
('SEC-005', 'Sensitive Data Export', 'Security', 'Critical', 'Large export of classified/sensitive data detected', 10, 30, 'SMS,Email,PagerDuty', 'CJIS,HIPAA,PCI-DSS,GDPR', 'All'),
('SEC-006', 'Unauthorized Schema Change', 'Security', 'High', 'DDL change by unauthorized principal', 15, 60, 'Email,SMS', 'SOX,PCI-DSS', 'All'),
('SEC-007', 'SA Account Activity', 'Security', 'High', 'Activity detected on SA or equivalent account', 10, 30, 'SMS,Email', 'CJIS,STIG', 'All'),
('SEC-008', 'Encryption Disabled', 'Security', 'Critical', 'TDE or connection encryption has been disabled', 5, 15, 'Phone,SMS,Email', 'CJIS,HIPAA,PCI-DSS', 'All'),
('SEC-009', 'Certificate Expiring', 'Security', 'High', 'SSL/TDE certificate expires within 30 days', 1440, 2880, 'Email', 'All', 'All'),
('SEC-010', 'Audit Trail Tampering', 'Security', 'Critical', 'Attempt to modify or delete audit records detected', 5, 10, 'Phone,SMS,Email,PagerDuty', 'All', 'All'),

-- AVAILABILITY ALERTS
('AVL-001', 'Server Offline', 'Availability', 'Critical', 'SQL Server instance is not responding', 5, 15, 'Phone,SMS,Email,PagerDuty', 'All', 'All'),
('AVL-002', 'Database Offline', 'Availability', 'Critical', 'Production database is offline or suspect', 5, 15, 'Phone,SMS,Email', 'All', 'All'),
('AVL-003', 'AG Failover', 'Availability', 'Critical', 'Availability Group automatic failover occurred', 5, 15, 'Phone,SMS,Email', 'All', 'All'),
('AVL-004', 'Cluster Node Down', 'Availability', 'Critical', 'Windows cluster node is offline', 10, 30, 'SMS,Email,PagerDuty', 'All', 'All'),
('AVL-005', 'Replication Failure', 'Availability', 'High', 'Transactional replication has stopped', 15, 60, 'SMS,Email', 'All', 'All'),
('AVL-006', 'Log Shipping Delayed', 'Availability', 'High', 'Log shipping restore delayed beyond threshold', 30, 120, 'Email', 'All', 'All'),
('AVL-007', 'Backup Failure', 'Availability', 'High', 'Scheduled backup job has failed', 30, 120, 'SMS,Email', 'CJIS,HIPAA,PCI-DSS', 'All'),
('AVL-008', 'Disk Space Critical', 'Availability', 'Critical', 'Disk space below 5% free', 15, 30, 'SMS,Email', 'All', 'All'),
('AVL-009', 'Transaction Log Full', 'Availability', 'Critical', 'Transaction log at 95%+ capacity', 10, 30, 'SMS,Email', 'All', 'All'),
('AVL-010', 'Memory Exhaustion', 'Availability', 'Critical', 'Available memory below critical threshold', 10, 30, 'SMS,Email', 'All', 'All'),

-- PERFORMANCE ALERTS
('PRF-001', 'CPU Critical', 'Performance', 'High', 'CPU utilization above 95% for 10+ minutes', 15, 60, 'SMS,Email', 'All', 'All'),
('PRF-002', 'Severe Blocking', 'Performance', 'High', 'Blocking chain affecting 10+ sessions', 10, 30, 'SMS,Email', 'All', 'All'),
('PRF-003', 'Deadlock Detected', 'Performance', 'Medium', 'Deadlock victim detected', 60, 240, 'Email', 'All', 'All'),
('PRF-004', 'Long Running Query', 'Performance', 'Medium', 'Query running longer than threshold', 30, 120, 'Email', 'All', 'All'),
('PRF-005', 'TempDB Contention', 'Performance', 'High', 'High TempDB allocation contention', 15, 60, 'SMS,Email', 'All', 'All'),

-- COMPLIANCE ALERTS
('CMP-001', 'Compliance Violation', 'Compliance', 'Critical', 'Action detected that violates compliance policy', 10, 30, 'Phone,SMS,Email', 'All', 'All'),
('CMP-002', 'Audit Gap Detected', 'Compliance', 'High', 'Gap in audit trail coverage detected', 30, 120, 'SMS,Email', 'CJIS,HIPAA,PCI-DSS,SOX', 'All'),
('CMP-003', 'Policy Override', 'Compliance', 'High', 'Security policy has been overridden', 15, 60, 'SMS,Email', 'All', 'All'),
('CMP-004', 'Retention Violation', 'Compliance', 'High', 'Data retention policy violation detected', 60, 240, 'Email', 'HIPAA,SOX,GDPR', 'All'),
('CMP-005', 'Access Review Overdue', 'Compliance', 'Medium', 'Quarterly access review not completed', 1440, 2880, 'Email', 'SOX,PCI-DSS', 'All'),

-- DATA PROTECTION ALERTS
('DPR-001', 'PII Access Anomaly', 'DataProtection', 'High', 'Unusual access pattern to PII data', 15, 60, 'SMS,Email', 'HIPAA,GDPR', 'Healthcare,Insurance,HR'),
('DPR-002', 'PHI Bulk Export', 'DataProtection', 'Critical', 'Large volume PHI export detected', 10, 30, 'Phone,SMS,Email', 'HIPAA', 'Healthcare'),
('DPR-003', 'CJI Access After Hours', 'DataProtection', 'High', 'Criminal Justice data accessed outside business hours', 15, 60, 'SMS,Email', 'CJIS', 'Law Enforcement'),
('DPR-004', 'Cardholder Data Access', 'DataProtection', 'High', 'Unusual cardholder data access pattern', 15, 60, 'SMS,Email', 'PCI-DSS', 'Banking,Retail'),
('DPR-005', 'Student Record Access', 'DataProtection', 'Medium', 'Bulk student record access detected', 30, 120, 'Email', 'FERPA', 'Education');

-- Alert History/Incident Log
CREATE TABLE alerts.IncidentLog (
    IncidentID BIGINT IDENTITY(1,1) PRIMARY KEY,
    AlertDefID INT FOREIGN KEY REFERENCES alerts.CriticalAlertDefinitions(AlertDefID),
    AlertCode NVARCHAR(20),
    IncidentTimestamp DATETIME2 DEFAULT GETDATE(),
    ServerName NVARCHAR(128),
    DatabaseName NVARCHAR(128),
    IncidentDetails NVARCHAR(MAX),
    AffectedObjects NVARCHAR(MAX),
    DetectedBy NVARCHAR(100), -- System, User, External
    Status NVARCHAR(20) DEFAULT 'Open', -- Open, Acknowledged, InProgress, Resolved, Escalated
    AcknowledgedBy NVARCHAR(128),
    AcknowledgedAt DATETIME2,
    AssignedTo NVARCHAR(128),
    EscalationLevel INT DEFAULT 0,
    EscalatedTo NVARCHAR(128),
    EscalatedAt DATETIME2,
    ResolutionSummary NVARCHAR(MAX),
    ResolvedBy NVARCHAR(128),
    ResolvedAt DATETIME2,
    RootCause NVARCHAR(500),
    PreventiveMeasures NVARCHAR(MAX),
    ResponseTime_Minutes INT,
    SLAMet BIT,
    INDEX IX_Incident_Time NONCLUSTERED (IncidentTimestamp),
    INDEX IX_Incident_Status NONCLUSTERED (Status)
);

-- ============================================================
-- LICENSING & SUBSCRIPTION MANAGEMENT
-- ============================================================
IF NOT EXISTS (SELECT 1 FROM sys.schemas WHERE name = 'licensing')
    EXEC('CREATE SCHEMA licensing');
GO

-- License Tiers
CREATE TABLE licensing.LicenseTiers (
    TierID INT IDENTITY(1,1) PRIMARY KEY,
    TierCode NVARCHAR(20) NOT NULL UNIQUE,
    TierName NVARCHAR(50) NOT NULL,
    Description NVARCHAR(500),
    MonthlyPrice DECIMAL(10,2),
    AnnualPrice DECIMAL(10,2),
    MaxServers INT,
    MaxDatabases INT,
    MaxUsers INT,
    Features NVARCHAR(MAX), -- JSON array of feature codes
    SupportLevel NVARCHAR(50), -- Community, Email, Priority, Premium, Dedicated
    SLAResponseHours INT,
    DataRetentionDays INT,
    IsActive BIT DEFAULT 1
);

-- Pre-populate License Tiers
INSERT INTO licensing.LicenseTiers 
(TierCode, TierName, Description, MonthlyPrice, AnnualPrice, MaxServers, MaxDatabases, MaxUsers, Features, SupportLevel, SLAResponseHours, DataRetentionDays)
VALUES
('FREE', 'Community', 'Free tier for evaluation and small environments', 0, 0, 2, 5, 2, 
 '["MONITORING_BASIC","ALERTS_BASIC","REPORTS_BASIC"]', 'Community', NULL, 7),

('STARTER', 'Starter', 'For small teams and single environments', 29.99, 299.99, 5, 20, 5,
 '["MONITORING_FULL","ALERTS_FULL","REPORTS_FULL","BACKUP_MONITOR","JOBS_MONITOR"]', 'Email', 24, 30),

('PROFESSIONAL', 'Professional', 'Full monitoring with self-healing capabilities', 99.99, 999.99, 15, 100, 15,
 '["MONITORING_FULL","ALERTS_FULL","REPORTS_FULL","BACKUP_MONITOR","JOBS_MONITOR","SELF_HEALING","HADR_MONITOR","COMPLIANCE_BASIC","MOBILE_APP"]', 'Priority', 8, 90),

('ENTERPRISE', 'Enterprise', 'Complete enterprise solution with compliance', 299.99, 2999.99, 50, 500, 50,
 '["MONITORING_FULL","ALERTS_FULL","REPORTS_FULL","BACKUP_MONITOR","JOBS_MONITOR","SELF_HEALING","HADR_MONITOR","COMPLIANCE_FULL","MOBILE_APP","API_ACCESS","SECURITY_AUDIT","ENCRYPTION_MONITOR","RBAC_ANALYSIS","CUSTOM_REPORTS"]', 'Premium', 4, 365),

('ULTIMATE', 'Ultimate', 'Unlimited everything with dedicated support', 599.99, 5999.99, -1, -1, -1,
 '["ALL_FEATURES","UNLIMITED_SERVERS","UNLIMITED_DATABASES","DEDICATED_SUPPORT","CUSTOM_DEVELOPMENT","ON_PREMISE_DEPLOYMENT","WHITE_LABEL","SLA_GUARANTEE"]', 'Dedicated', 1, 730),

('GOV', 'Government', 'CJIS/FedRAMP compliant deployment', NULL, NULL, -1, -1, -1,
 '["ALL_FEATURES","CJIS_COMPLIANT","FEDRAMP_READY","FIPS_ENCRYPTION","GOVT_SLA"]', 'Dedicated', 1, 2555);

-- Feature Catalog
CREATE TABLE licensing.FeatureCatalog (
    FeatureID INT IDENTITY(1,1) PRIMARY KEY,
    FeatureCode NVARCHAR(50) NOT NULL UNIQUE,
    FeatureName NVARCHAR(100) NOT NULL,
    FeatureCategory NVARCHAR(50),
    Description NVARCHAR(500),
    MinimumTier NVARCHAR(20),
    IsAddon BIT DEFAULT 0,
    AddonPrice DECIMAL(10,2),
    IsActive BIT DEFAULT 1
);

-- Pre-populate Features
INSERT INTO licensing.FeatureCatalog (FeatureCode, FeatureName, FeatureCategory, Description, MinimumTier, IsAddon, AddonPrice)
VALUES
-- Monitoring Features
('MONITORING_BASIC', 'Basic Monitoring', 'Monitoring', 'Server health, basic metrics', 'FREE', 0, NULL),
('MONITORING_FULL', 'Full Monitoring', 'Monitoring', 'Complete performance monitoring with history', 'STARTER', 0, NULL),
('REALTIME_METRICS', 'Real-Time Metrics', 'Monitoring', 'Sub-second metric updates', 'PROFESSIONAL', 0, NULL),

-- Alert Features
('ALERTS_BASIC', 'Basic Alerts', 'Alerts', 'Email alerts for critical issues', 'FREE', 0, NULL),
('ALERTS_FULL', 'Full Alerts', 'Alerts', 'Multi-channel alerts with escalation', 'STARTER', 0, NULL),
('ALERTS_SMS', 'SMS Alerts', 'Alerts', 'SMS notification capability', 'PROFESSIONAL', 1, 9.99),
('ALERTS_PAGERDUTY', 'PagerDuty Integration', 'Alerts', 'PagerDuty alert routing', 'PROFESSIONAL', 1, 14.99),

-- Self-Healing
('SELF_HEALING', 'Self-Healing Engine', 'Automation', 'Automatic issue remediation', 'PROFESSIONAL', 0, NULL),
('AUTO_BACKUP', 'Auto-Backup on Failure', 'Automation', 'Automatic backup on missed schedule', 'PROFESSIONAL', 0, NULL),
('AUTO_INDEX', 'Auto Index Maintenance', 'Automation', 'Automatic index optimization', 'PROFESSIONAL', 0, NULL),

-- Compliance
('COMPLIANCE_BASIC', 'Basic Compliance', 'Compliance', 'Compliance dashboards and basic reports', 'PROFESSIONAL', 0, NULL),
('COMPLIANCE_FULL', 'Full Compliance Suite', 'Compliance', 'Complete compliance with evidence generation', 'ENTERPRISE', 0, NULL),
('CJIS_MODULE', 'CJIS Compliance Module', 'Compliance', 'CJIS-specific controls and reporting', 'ENTERPRISE', 1, 49.99),
('HIPAA_MODULE', 'HIPAA Compliance Module', 'Compliance', 'HIPAA-specific controls and reporting', 'ENTERPRISE', 1, 49.99),
('PCI_MODULE', 'PCI-DSS Compliance Module', 'Compliance', 'PCI-DSS specific controls and reporting', 'ENTERPRISE', 1, 49.99),

-- Security
('SECURITY_AUDIT', 'Security Audit', 'Security', 'Complete security auditing', 'ENTERPRISE', 0, NULL),
('ENCRYPTION_MONITOR', 'Encryption Monitoring', 'Security', 'TDE/TLS/backup encryption monitoring', 'ENTERPRISE', 0, NULL),
('RBAC_ANALYSIS', 'RBAC Analysis', 'Security', 'Role-based access control analysis', 'ENTERPRISE', 0, NULL),
('THREAT_DETECTION', 'Threat Detection', 'Security', 'AI-powered threat detection', 'ULTIMATE', 0, NULL),

-- Mobile & API
('MOBILE_APP', 'Mobile Application', 'Platform', 'iOS and Android mobile apps', 'PROFESSIONAL', 0, NULL),
('API_ACCESS', 'API Access', 'Platform', 'REST API access for integrations', 'ENTERPRISE', 0, NULL),

-- Reports
('REPORTS_BASIC', 'Basic Reports', 'Reports', 'Standard operational reports', 'FREE', 0, NULL),
('REPORTS_FULL', 'Full Reports', 'Reports', 'Complete report library', 'STARTER', 0, NULL),
('CUSTOM_REPORTS', 'Custom Reports', 'Reports', 'Custom report builder', 'ENTERPRISE', 0, NULL),
('EXECUTIVE_DASHBOARD', 'Executive Dashboard', 'Reports', 'C-level executive dashboards', 'ENTERPRISE', 1, 29.99);

-- Customer Licenses
CREATE TABLE licensing.CustomerLicenses (
    LicenseID INT IDENTITY(1,1) PRIMARY KEY,
    CustomerID INT,
    CustomerName NVARCHAR(200) NOT NULL,
    TierID INT FOREIGN KEY REFERENCES licensing.LicenseTiers(TierID),
    LicenseKey NVARCHAR(100) NOT NULL UNIQUE,
    ActivatedDate DATETIME2,
    ExpirationDate DATETIME2,
    BillingCycle NVARCHAR(20), -- Monthly, Annual, Perpetual
    Status NVARCHAR(20) DEFAULT 'Active', -- Active, Expired, Suspended, Cancelled
    ContactName NVARCHAR(100),
    ContactEmail NVARCHAR(200),
    Industry NVARCHAR(100),
    ServerCount INT DEFAULT 0,
    DatabaseCount INT DEFAULT 0,
    UserCount INT DEFAULT 0,
    AddOns NVARCHAR(MAX), -- JSON array of addon feature codes
    Notes NVARCHAR(MAX),
    CreatedDate DATETIME2 DEFAULT GETDATE()
);

PRINT '✅ Comprehensive Audit & Security Schema Created';
PRINT '✅ Retention Policies Populated (CJIS, HIPAA, PCI-DSS, SOX, GDPR, FERPA)';
PRINT '✅ Critical Alert Definitions Created (30+ industry-standard alerts)';
PRINT '✅ Licensing Tiers Configured (Free to Ultimate + Government)';
